/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
*** All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Revoke mosaic.
**/
public class MosaicSupplyRevocationTransactionBuilder extends TransactionBuilder implements Serializer {

    /** Mosaic supply revocation transaction body. **/
    private final MosaicSupplyRevocationTransactionBodyBuilder mosaicSupplyRevocationTransactionBody;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MosaicSupplyRevocationTransactionBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.mosaicSupplyRevocationTransactionBody = MosaicSupplyRevocationTransactionBodyBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of MosaicSupplyRevocationTransactionBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicSupplyRevocationTransactionBuilder.
     */
    public static MosaicSupplyRevocationTransactionBuilder loadFromBinary(DataInputStream stream) {
        return new MosaicSupplyRevocationTransactionBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param signature Entity's signature generated by the signing account..
    * @param signerPublicKey Public key of the signer of the entity..
    * @param version Version of this structure..
    * @param network Network on which this entity was created..
    * @param type Transaction type.
    * @param fee Transaction fee.
    * @param deadline Transaction deadline.
    * @param sourceAddress Address from which tokens should be revoked..
    * @param mosaic Revoked mosaic and amount..
    */
    protected MosaicSupplyRevocationTransactionBuilder(SignatureDto signature, PublicKeyDto signerPublicKey, byte version, NetworkTypeDto network, TransactionTypeDto type, AmountDto fee, TimestampDto deadline, UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        super(signature, signerPublicKey, version, network, type, fee, deadline);
        GeneratorUtils.notNull(signature, "signature is null");
        GeneratorUtils.notNull(signerPublicKey, "signerPublicKey is null");
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(network, "network is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(fee, "fee is null");
        GeneratorUtils.notNull(deadline, "deadline is null");
        GeneratorUtils.notNull(sourceAddress, "sourceAddress is null");
        GeneratorUtils.notNull(mosaic, "mosaic is null");
        this.mosaicSupplyRevocationTransactionBody = new MosaicSupplyRevocationTransactionBodyBuilder(sourceAddress, mosaic);
    }
    
    /**
     * Creates an instance of MosaicSupplyRevocationTransactionBuilder.
     *
     * @param signature Entity's signature generated by the signing account..
     * @param signerPublicKey Public key of the signer of the entity..
     * @param version Version of this structure..
     * @param network Network on which this entity was created..
     * @param type Transaction type.
     * @param fee Transaction fee.
     * @param deadline Transaction deadline.
     * @param sourceAddress Address from which tokens should be revoked..
     * @param mosaic Revoked mosaic and amount..
     * @return Instance of MosaicSupplyRevocationTransactionBuilder.
     */
    public static MosaicSupplyRevocationTransactionBuilder create(SignatureDto signature, PublicKeyDto signerPublicKey, byte version, NetworkTypeDto network, TransactionTypeDto type, AmountDto fee, TimestampDto deadline, UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        return new MosaicSupplyRevocationTransactionBuilder(signature, signerPublicKey, version, network, type, fee, deadline, sourceAddress, mosaic);
    }

    /**
     * Gets Address from which tokens should be revoked..
     *
     * @return Address from which tokens should be revoked..
     */
    public UnresolvedAddressDto getSourceAddress() {
        return this.mosaicSupplyRevocationTransactionBody.getSourceAddress();
    }

    /**
     * Gets Revoked mosaic and amount..
     *
     * @return Revoked mosaic and amount..
     */
    public UnresolvedMosaicBuilder getMosaic() {
        return this.mosaicSupplyRevocationTransactionBody.getMosaic();
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += this.mosaicSupplyRevocationTransactionBody.getSize();
        return size;
    }

    /**
     * Gets the body builder of the object.
     *
     * @return Body builder.
     */
    @Override
    public MosaicSupplyRevocationTransactionBodyBuilder getBody() {
        return this.mosaicSupplyRevocationTransactionBody;
    }


    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            GeneratorUtils.writeEntity(dataOutputStream, this.mosaicSupplyRevocationTransactionBody);
        });
    }
}

