/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
*** All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Shared content between MosaicSupplyRevocationTransaction and EmbeddedMosaicSupplyRevocationTransaction.
**/
public class MosaicSupplyRevocationTransactionBodyBuilder implements Serializer {

    /** Address from which tokens should be revoked.. **/
    private final UnresolvedAddressDto sourceAddress;

    /** Revoked mosaic and amount.. **/
    private final UnresolvedMosaicBuilder mosaic;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MosaicSupplyRevocationTransactionBodyBuilder(DataInputStream stream) {
        try {
            this.sourceAddress = UnresolvedAddressDto.loadFromBinary(stream);
            this.mosaic = UnresolvedMosaicBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of MosaicSupplyRevocationTransactionBodyBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicSupplyRevocationTransactionBodyBuilder.
     */
    public static MosaicSupplyRevocationTransactionBodyBuilder loadFromBinary(DataInputStream stream) {
        return new MosaicSupplyRevocationTransactionBodyBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param sourceAddress Address from which tokens should be revoked..
    * @param mosaic Revoked mosaic and amount..
    */
    protected MosaicSupplyRevocationTransactionBodyBuilder(UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        GeneratorUtils.notNull(sourceAddress, "sourceAddress is null");
        GeneratorUtils.notNull(mosaic, "mosaic is null");
        this.sourceAddress = sourceAddress;
        this.mosaic = mosaic;
    }
    
    /**
     * Creates an instance of MosaicSupplyRevocationTransactionBodyBuilder.
     *
     * @param sourceAddress Address from which tokens should be revoked..
     * @param mosaic Revoked mosaic and amount..
     * @return Instance of MosaicSupplyRevocationTransactionBodyBuilder.
     */
    public static MosaicSupplyRevocationTransactionBodyBuilder create(UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        return new MosaicSupplyRevocationTransactionBodyBuilder(sourceAddress, mosaic);
    }

    /**
     * Gets Address from which tokens should be revoked..
     *
     * @return Address from which tokens should be revoked..
     */
    public UnresolvedAddressDto getSourceAddress() {
        return this.sourceAddress;
    }

    /**
     * Gets Revoked mosaic and amount..
     *
     * @return Revoked mosaic and amount..
     */
    public UnresolvedMosaicBuilder getMosaic() {
        return this.mosaic;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.sourceAddress.getSize();
        size += this.mosaic.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.sourceAddress);
            GeneratorUtils.writeEntity(dataOutputStream, this.mosaic);
        });
    }
}

