/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
*** All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for an importance block header
**/
public class ImportanceBlockHeaderBuilder extends BlockHeaderBuilder implements Serializer {

    /** Importance block footer. **/
    private final ImportanceBlockFooterBuilder importanceBlockFooter;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected ImportanceBlockHeaderBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.importanceBlockFooter = ImportanceBlockFooterBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of ImportanceBlockHeaderBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of ImportanceBlockHeaderBuilder.
     */
    public static ImportanceBlockHeaderBuilder loadFromBinary(DataInputStream stream) {
        return new ImportanceBlockHeaderBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param signature Entity's signature generated by the signing account..
    * @param signerPublicKey Public key of the signer of the entity..
    * @param version Version of this structure..
    * @param network Network on which this entity was created..
    * @param type Block type.
    * @param height Block height.
    * @param timestamp Number of milliseconds elapsed since creation of nemesis block.
    * @param difficulty Block difficulty.
    * @param generationHashProof Generation hash proof.
    * @param previousBlockHash Previous block hash.
    * @param transactionsHash Hash of the transactions in this block.
    * @param receiptsHash Hash of the receipts generated by this block.
    * @param stateHash Hash of the global chain state at this block.
    * @param beneficiaryAddress Beneficiary address designated by harvester.
    * @param feeMultiplier Fee multiplier applied to block transactions.
    * @param votingEligibleAccountsCount Number of voting eligible accounts.
    * @param harvestingEligibleAccountsCount Number of harvesting eligible accounts.
    * @param totalVotingBalance Total balance eligible for voting.
    * @param previousImportanceBlockHash Previous importance block hash.
    */
    protected ImportanceBlockHeaderBuilder(SignatureDto signature, PublicKeyDto signerPublicKey, byte version, NetworkTypeDto network, BlockTypeDto type, HeightDto height, TimestampDto timestamp, DifficultyDto difficulty, VrfProofBuilder generationHashProof, Hash256Dto previousBlockHash, Hash256Dto transactionsHash, Hash256Dto receiptsHash, Hash256Dto stateHash, AddressDto beneficiaryAddress, BlockFeeMultiplierDto feeMultiplier, int votingEligibleAccountsCount, long harvestingEligibleAccountsCount, AmountDto totalVotingBalance, Hash256Dto previousImportanceBlockHash) {
        super(signature, signerPublicKey, version, network, type, height, timestamp, difficulty, generationHashProof, previousBlockHash, transactionsHash, receiptsHash, stateHash, beneficiaryAddress, feeMultiplier);
        GeneratorUtils.notNull(signature, "signature is null");
        GeneratorUtils.notNull(signerPublicKey, "signerPublicKey is null");
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(network, "network is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(height, "height is null");
        GeneratorUtils.notNull(timestamp, "timestamp is null");
        GeneratorUtils.notNull(difficulty, "difficulty is null");
        GeneratorUtils.notNull(generationHashProof, "generationHashProof is null");
        GeneratorUtils.notNull(previousBlockHash, "previousBlockHash is null");
        GeneratorUtils.notNull(transactionsHash, "transactionsHash is null");
        GeneratorUtils.notNull(receiptsHash, "receiptsHash is null");
        GeneratorUtils.notNull(stateHash, "stateHash is null");
        GeneratorUtils.notNull(beneficiaryAddress, "beneficiaryAddress is null");
        GeneratorUtils.notNull(feeMultiplier, "feeMultiplier is null");
        GeneratorUtils.notNull(votingEligibleAccountsCount, "votingEligibleAccountsCount is null");
        GeneratorUtils.notNull(harvestingEligibleAccountsCount, "harvestingEligibleAccountsCount is null");
        GeneratorUtils.notNull(totalVotingBalance, "totalVotingBalance is null");
        GeneratorUtils.notNull(previousImportanceBlockHash, "previousImportanceBlockHash is null");
        this.importanceBlockFooter = new ImportanceBlockFooterBuilder(votingEligibleAccountsCount, harvestingEligibleAccountsCount, totalVotingBalance, previousImportanceBlockHash);
    }
    
    /**
     * Creates an instance of ImportanceBlockHeaderBuilder.
     *
     * @param signature Entity's signature generated by the signing account..
     * @param signerPublicKey Public key of the signer of the entity..
     * @param version Version of this structure..
     * @param network Network on which this entity was created..
     * @param type Block type.
     * @param height Block height.
     * @param timestamp Number of milliseconds elapsed since creation of nemesis block.
     * @param difficulty Block difficulty.
     * @param generationHashProof Generation hash proof.
     * @param previousBlockHash Previous block hash.
     * @param transactionsHash Hash of the transactions in this block.
     * @param receiptsHash Hash of the receipts generated by this block.
     * @param stateHash Hash of the global chain state at this block.
     * @param beneficiaryAddress Beneficiary address designated by harvester.
     * @param feeMultiplier Fee multiplier applied to block transactions.
     * @param votingEligibleAccountsCount Number of voting eligible accounts.
     * @param harvestingEligibleAccountsCount Number of harvesting eligible accounts.
     * @param totalVotingBalance Total balance eligible for voting.
     * @param previousImportanceBlockHash Previous importance block hash.
     * @return Instance of ImportanceBlockHeaderBuilder.
     */
    public static ImportanceBlockHeaderBuilder create(SignatureDto signature, PublicKeyDto signerPublicKey, byte version, NetworkTypeDto network, BlockTypeDto type, HeightDto height, TimestampDto timestamp, DifficultyDto difficulty, VrfProofBuilder generationHashProof, Hash256Dto previousBlockHash, Hash256Dto transactionsHash, Hash256Dto receiptsHash, Hash256Dto stateHash, AddressDto beneficiaryAddress, BlockFeeMultiplierDto feeMultiplier, int votingEligibleAccountsCount, long harvestingEligibleAccountsCount, AmountDto totalVotingBalance, Hash256Dto previousImportanceBlockHash) {
        return new ImportanceBlockHeaderBuilder(signature, signerPublicKey, version, network, type, height, timestamp, difficulty, generationHashProof, previousBlockHash, transactionsHash, receiptsHash, stateHash, beneficiaryAddress, feeMultiplier, votingEligibleAccountsCount, harvestingEligibleAccountsCount, totalVotingBalance, previousImportanceBlockHash);
    }

    /**
     * Gets number of voting eligible accounts.
     *
     * @return Number of voting eligible accounts.
     */
    public int getVotingEligibleAccountsCount() {
        return this.importanceBlockFooter.getVotingEligibleAccountsCount();
    }

    /**
     * Gets number of harvesting eligible accounts.
     *
     * @return Number of harvesting eligible accounts.
     */
    public long getHarvestingEligibleAccountsCount() {
        return this.importanceBlockFooter.getHarvestingEligibleAccountsCount();
    }

    /**
     * Gets total balance eligible for voting.
     *
     * @return Total balance eligible for voting.
     */
    public AmountDto getTotalVotingBalance() {
        return this.importanceBlockFooter.getTotalVotingBalance();
    }

    /**
     * Gets previous importance block hash.
     *
     * @return Previous importance block hash.
     */
    public Hash256Dto getPreviousImportanceBlockHash() {
        return this.importanceBlockFooter.getPreviousImportanceBlockHash();
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += this.importanceBlockFooter.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            GeneratorUtils.writeEntity(dataOutputStream, this.importanceBlockFooter);
        });
    }
}

