/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
*** All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Embedded version of MosaicSupplyRevocationTransaction.
**/
public class EmbeddedMosaicSupplyRevocationTransactionBuilder extends EmbeddedTransactionBuilder implements Serializer {

    /** Mosaic supply revocation transaction body. **/
    private final MosaicSupplyRevocationTransactionBodyBuilder mosaicSupplyRevocationTransactionBody;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected EmbeddedMosaicSupplyRevocationTransactionBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.mosaicSupplyRevocationTransactionBody = MosaicSupplyRevocationTransactionBodyBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of EmbeddedMosaicSupplyRevocationTransactionBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of EmbeddedMosaicSupplyRevocationTransactionBuilder.
     */
    public static EmbeddedMosaicSupplyRevocationTransactionBuilder loadFromBinary(DataInputStream stream) {
        return new EmbeddedMosaicSupplyRevocationTransactionBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param signerPublicKey Public key of the signer of the entity..
    * @param version Version of this structure..
    * @param network Network on which this entity was created..
    * @param type Transaction type.
    * @param sourceAddress Address from which tokens should be revoked..
    * @param mosaic Revoked mosaic and amount..
    */
    protected EmbeddedMosaicSupplyRevocationTransactionBuilder(PublicKeyDto signerPublicKey, byte version, NetworkTypeDto network, TransactionTypeDto type, UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        super(signerPublicKey, version, network, type);
        GeneratorUtils.notNull(signerPublicKey, "signerPublicKey is null");
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(network, "network is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(sourceAddress, "sourceAddress is null");
        GeneratorUtils.notNull(mosaic, "mosaic is null");
        this.mosaicSupplyRevocationTransactionBody = new MosaicSupplyRevocationTransactionBodyBuilder(sourceAddress, mosaic);
    }
    
    /**
     * Creates an instance of EmbeddedMosaicSupplyRevocationTransactionBuilder.
     *
     * @param signerPublicKey Public key of the signer of the entity..
     * @param version Version of this structure..
     * @param network Network on which this entity was created..
     * @param type Transaction type.
     * @param sourceAddress Address from which tokens should be revoked..
     * @param mosaic Revoked mosaic and amount..
     * @return Instance of EmbeddedMosaicSupplyRevocationTransactionBuilder.
     */
    public static EmbeddedMosaicSupplyRevocationTransactionBuilder create(PublicKeyDto signerPublicKey, byte version, NetworkTypeDto network, TransactionTypeDto type, UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        return new EmbeddedMosaicSupplyRevocationTransactionBuilder(signerPublicKey, version, network, type, sourceAddress, mosaic);
    }

    /**
     * Gets Address from which tokens should be revoked..
     *
     * @return Address from which tokens should be revoked..
     */
    public UnresolvedAddressDto getSourceAddress() {
        return this.mosaicSupplyRevocationTransactionBody.getSourceAddress();
    }

    /**
     * Gets Revoked mosaic and amount..
     *
     * @return Revoked mosaic and amount..
     */
    public UnresolvedMosaicBuilder getMosaic() {
        return this.mosaicSupplyRevocationTransactionBody.getMosaic();
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += this.mosaicSupplyRevocationTransactionBody.getSize();
        return size;
    }

    /**
     * Gets the body builder of the object.
     *
     * @return Body builder.
     */
    @Override
    public MosaicSupplyRevocationTransactionBodyBuilder getBody() {
        return this.mosaicSupplyRevocationTransactionBody;
    }


    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            GeneratorUtils.writeEntity(dataOutputStream, this.mosaicSupplyRevocationTransactionBody);
        });
    }
}

