package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Pinned voting key
**/
public class PinnedVotingKeyBuilder implements Serializer {

    /** Voting key. **/
    private final VotingKeyDto votingKey;

    /** Start finalization epoch. **/
    private final FinalizationEpochDto startEpoch;

    /** End finalization epoch. **/
    private final FinalizationEpochDto endEpoch;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected PinnedVotingKeyBuilder(DataInputStream stream) {
        try {
            this.votingKey = VotingKeyDto.loadFromBinary(stream);
            this.startEpoch = FinalizationEpochDto.loadFromBinary(stream);
            this.endEpoch = FinalizationEpochDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of PinnedVotingKeyBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of PinnedVotingKeyBuilder.
     */
    public static PinnedVotingKeyBuilder loadFromBinary(DataInputStream stream) {
        return new PinnedVotingKeyBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param votingKey Voting key.
    * @param startEpoch Start finalization epoch.
    * @param endEpoch End finalization epoch.
    */
    protected PinnedVotingKeyBuilder(VotingKeyDto votingKey, FinalizationEpochDto startEpoch, FinalizationEpochDto endEpoch) {
        GeneratorUtils.notNull(votingKey, "votingKey is null");
        GeneratorUtils.notNull(startEpoch, "startEpoch is null");
        GeneratorUtils.notNull(endEpoch, "endEpoch is null");
        this.votingKey = votingKey;
        this.startEpoch = startEpoch;
        this.endEpoch = endEpoch;
    }
    
    /**
     * Creates an instance of PinnedVotingKeyBuilder.
     *
     * @param votingKey Voting key.
     * @param startEpoch Start finalization epoch.
     * @param endEpoch End finalization epoch.
     * @return Instance of PinnedVotingKeyBuilder.
     */
    public static PinnedVotingKeyBuilder create(VotingKeyDto votingKey, FinalizationEpochDto startEpoch, FinalizationEpochDto endEpoch) {
        return new PinnedVotingKeyBuilder(votingKey, startEpoch, endEpoch);
    }

    /**
     * Gets voting key.
     *
     * @return Voting key.
     */
    public VotingKeyDto getVotingKey() {
        return this.votingKey;
    }

    /**
     * Gets start finalization epoch.
     *
     * @return Start finalization epoch.
     */
    public FinalizationEpochDto getStartEpoch() {
        return this.startEpoch;
    }

    /**
     * Gets end finalization epoch.
     *
     * @return End finalization epoch.
     */
    public FinalizationEpochDto getEndEpoch() {
        return this.endEpoch;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.votingKey.getSize();
        size += this.startEpoch.getSize();
        size += this.endEpoch.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.votingKey);
            GeneratorUtils.writeEntity(dataOutputStream, this.startEpoch);
            GeneratorUtils.writeEntity(dataOutputStream, this.endEpoch);
        });
    }
}

