package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a non-embedded mosaic supply revocation transaction
**/
public class MosaicSupplyRevocationTransactionBuilder extends TransactionBuilder implements Serializer {

    /** Mosaic supply revocation transaction body. **/
    private final MosaicSupplyRevocationTransactionBodyBuilder mosaicSupplyRevocationTransactionBody;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MosaicSupplyRevocationTransactionBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.mosaicSupplyRevocationTransactionBody = MosaicSupplyRevocationTransactionBodyBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of MosaicSupplyRevocationTransactionBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicSupplyRevocationTransactionBuilder.
     */
    public static MosaicSupplyRevocationTransactionBuilder loadFromBinary(DataInputStream stream) {
        return new MosaicSupplyRevocationTransactionBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param signature Entity signature.
    * @param signerPublicKey Entity signer's public key.
    * @param version Entity version.
    * @param network Entity network.
    * @param type Entity type.
    * @param fee Transaction fee.
    * @param deadline Transaction deadline.
    * @param sourceAddress Address from which tokens should be revoked.
    * @param mosaic Revoked mosaic and amount.
    */
    protected MosaicSupplyRevocationTransactionBuilder(SignatureDto signature, KeyDto signerPublicKey, byte version, NetworkTypeDto network, EntityTypeDto type, AmountDto fee, TimestampDto deadline, UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        super(signature, signerPublicKey, version, network, type, fee, deadline);
        GeneratorUtils.notNull(signature, "signature is null");
        GeneratorUtils.notNull(signerPublicKey, "signerPublicKey is null");
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(network, "network is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(fee, "fee is null");
        GeneratorUtils.notNull(deadline, "deadline is null");
        GeneratorUtils.notNull(sourceAddress, "sourceAddress is null");
        GeneratorUtils.notNull(mosaic, "mosaic is null");
        this.mosaicSupplyRevocationTransactionBody = new MosaicSupplyRevocationTransactionBodyBuilder(sourceAddress, mosaic);
    }
    
    /**
     * Creates an instance of MosaicSupplyRevocationTransactionBuilder.
     *
     * @param signature Entity signature.
     * @param signerPublicKey Entity signer's public key.
     * @param version Entity version.
     * @param network Entity network.
     * @param type Entity type.
     * @param fee Transaction fee.
     * @param deadline Transaction deadline.
     * @param sourceAddress Address from which tokens should be revoked.
     * @param mosaic Revoked mosaic and amount.
     * @return Instance of MosaicSupplyRevocationTransactionBuilder.
     */
    public static MosaicSupplyRevocationTransactionBuilder create(SignatureDto signature, KeyDto signerPublicKey, byte version, NetworkTypeDto network, EntityTypeDto type, AmountDto fee, TimestampDto deadline, UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        return new MosaicSupplyRevocationTransactionBuilder(signature, signerPublicKey, version, network, type, fee, deadline, sourceAddress, mosaic);
    }

    /**
     * Gets address from which tokens should be revoked.
     *
     * @return Address from which tokens should be revoked.
     */
    public UnresolvedAddressDto getSourceAddress() {
        return this.mosaicSupplyRevocationTransactionBody.getSourceAddress();
    }

    /**
     * Gets revoked mosaic and amount.
     *
     * @return Revoked mosaic and amount.
     */
    public UnresolvedMosaicBuilder getMosaic() {
        return this.mosaicSupplyRevocationTransactionBody.getMosaic();
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += this.mosaicSupplyRevocationTransactionBody.getSize();
        return size;
    }

    /**
     * Gets the body builder of the object.
     *
     * @return Body builder.
     */
    @Override
    public MosaicSupplyRevocationTransactionBodyBuilder getBody() {
        return this.mosaicSupplyRevocationTransactionBody;
    }


    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            GeneratorUtils.writeEntity(dataOutputStream, this.mosaicSupplyRevocationTransactionBody);
        });
    }
}

