package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a mosaic restriction
**/
public class MosaicGlobalRestrictionEntryBuilder implements Serializer {

    /** Identifier of the mosaic to which the restriction applies. **/
    private final MosaicIdDto mosaicId;

    /** Global key value restriction set. **/
    private final GlobalKeyValueSetBuilder keyPairs;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MosaicGlobalRestrictionEntryBuilder(DataInputStream stream) {
        try {
            this.mosaicId = MosaicIdDto.loadFromBinary(stream);
            this.keyPairs = GlobalKeyValueSetBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of MosaicGlobalRestrictionEntryBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicGlobalRestrictionEntryBuilder.
     */
    public static MosaicGlobalRestrictionEntryBuilder loadFromBinary(DataInputStream stream) {
        return new MosaicGlobalRestrictionEntryBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param mosaicId Identifier of the mosaic to which the restriction applies.
    * @param keyPairs Global key value restriction set.
    */
    protected MosaicGlobalRestrictionEntryBuilder(MosaicIdDto mosaicId, GlobalKeyValueSetBuilder keyPairs) {
        GeneratorUtils.notNull(mosaicId, "mosaicId is null");
        GeneratorUtils.notNull(keyPairs, "keyPairs is null");
        this.mosaicId = mosaicId;
        this.keyPairs = keyPairs;
    }
    
    /**
     * Creates an instance of MosaicGlobalRestrictionEntryBuilder.
     *
     * @param mosaicId Identifier of the mosaic to which the restriction applies.
     * @param keyPairs Global key value restriction set.
     * @return Instance of MosaicGlobalRestrictionEntryBuilder.
     */
    public static MosaicGlobalRestrictionEntryBuilder create(MosaicIdDto mosaicId, GlobalKeyValueSetBuilder keyPairs) {
        return new MosaicGlobalRestrictionEntryBuilder(mosaicId, keyPairs);
    }

    /**
     * Gets identifier of the mosaic to which the restriction applies.
     *
     * @return Identifier of the mosaic to which the restriction applies.
     */
    public MosaicIdDto getMosaicId() {
        return this.mosaicId;
    }

    /**
     * Gets global key value restriction set.
     *
     * @return Global key value restriction set.
     */
    public GlobalKeyValueSetBuilder getKeyPairs() {
        return this.keyPairs;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.mosaicId.getSize();
        size += this.keyPairs.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.mosaicId);
            GeneratorUtils.writeEntity(dataOutputStream, this.keyPairs);
        });
    }
}

