package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a nemesis block header
**/
public class NemesisBlockHeaderBuilder extends BlockHeaderBuilder implements Serializer {

    /** Importance block footer. **/
    private final ImportanceBlockFooterBuilder importanceBlockFooter;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected NemesisBlockHeaderBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.importanceBlockFooter = ImportanceBlockFooterBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of NemesisBlockHeaderBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of NemesisBlockHeaderBuilder.
     */
    public static NemesisBlockHeaderBuilder loadFromBinary(DataInputStream stream) {
        return new NemesisBlockHeaderBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param signature Entity signature.
    * @param signerPublicKey Entity signer's public key.
    * @param version Entity version.
    * @param network Entity network.
    * @param type Entity type.
    * @param height Block height.
    * @param timestamp Number of milliseconds elapsed since creation of nemesis block.
    * @param difficulty Block difficulty.
    * @param generationHashProof Generation hash proof.
    * @param previousBlockHash Previous block hash.
    * @param transactionsHash Hash of the transactions in this block.
    * @param receiptsHash Hash of the receipts generated by this block.
    * @param stateHash Hash of the global chain state at this block.
    * @param beneficiaryAddress Beneficiary address designated by harvester.
    * @param feeMultiplier Fee multiplier applied to block transactions.
    * @param votingEligibleAccountsCount Number of voting eligible accounts.
    * @param harvestingEligibleAccountsCount Number of harvesting eligible accounts.
    * @param totalVotingBalance Total balance eligible for voting.
    * @param previousImportanceBlockHash Previous importance block hash.
    */
    protected NemesisBlockHeaderBuilder(SignatureDto signature, KeyDto signerPublicKey, byte version, NetworkTypeDto network, EntityTypeDto type, HeightDto height, TimestampDto timestamp, DifficultyDto difficulty, VrfProofBuilder generationHashProof, Hash256Dto previousBlockHash, Hash256Dto transactionsHash, Hash256Dto receiptsHash, Hash256Dto stateHash, AddressDto beneficiaryAddress, BlockFeeMultiplierDto feeMultiplier, int votingEligibleAccountsCount, long harvestingEligibleAccountsCount, AmountDto totalVotingBalance, Hash256Dto previousImportanceBlockHash) {
        super(signature, signerPublicKey, version, network, type, height, timestamp, difficulty, generationHashProof, previousBlockHash, transactionsHash, receiptsHash, stateHash, beneficiaryAddress, feeMultiplier);
        GeneratorUtils.notNull(signature, "signature is null");
        GeneratorUtils.notNull(signerPublicKey, "signerPublicKey is null");
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(network, "network is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(height, "height is null");
        GeneratorUtils.notNull(timestamp, "timestamp is null");
        GeneratorUtils.notNull(difficulty, "difficulty is null");
        GeneratorUtils.notNull(generationHashProof, "generationHashProof is null");
        GeneratorUtils.notNull(previousBlockHash, "previousBlockHash is null");
        GeneratorUtils.notNull(transactionsHash, "transactionsHash is null");
        GeneratorUtils.notNull(receiptsHash, "receiptsHash is null");
        GeneratorUtils.notNull(stateHash, "stateHash is null");
        GeneratorUtils.notNull(beneficiaryAddress, "beneficiaryAddress is null");
        GeneratorUtils.notNull(feeMultiplier, "feeMultiplier is null");
        GeneratorUtils.notNull(votingEligibleAccountsCount, "votingEligibleAccountsCount is null");
        GeneratorUtils.notNull(harvestingEligibleAccountsCount, "harvestingEligibleAccountsCount is null");
        GeneratorUtils.notNull(totalVotingBalance, "totalVotingBalance is null");
        GeneratorUtils.notNull(previousImportanceBlockHash, "previousImportanceBlockHash is null");
        this.importanceBlockFooter = new ImportanceBlockFooterBuilder(votingEligibleAccountsCount, harvestingEligibleAccountsCount, totalVotingBalance, previousImportanceBlockHash);
    }
    
    /**
     * Creates an instance of NemesisBlockHeaderBuilder.
     *
     * @param signature Entity signature.
     * @param signerPublicKey Entity signer's public key.
     * @param version Entity version.
     * @param network Entity network.
     * @param type Entity type.
     * @param height Block height.
     * @param timestamp Number of milliseconds elapsed since creation of nemesis block.
     * @param difficulty Block difficulty.
     * @param generationHashProof Generation hash proof.
     * @param previousBlockHash Previous block hash.
     * @param transactionsHash Hash of the transactions in this block.
     * @param receiptsHash Hash of the receipts generated by this block.
     * @param stateHash Hash of the global chain state at this block.
     * @param beneficiaryAddress Beneficiary address designated by harvester.
     * @param feeMultiplier Fee multiplier applied to block transactions.
     * @param votingEligibleAccountsCount Number of voting eligible accounts.
     * @param harvestingEligibleAccountsCount Number of harvesting eligible accounts.
     * @param totalVotingBalance Total balance eligible for voting.
     * @param previousImportanceBlockHash Previous importance block hash.
     * @return Instance of NemesisBlockHeaderBuilder.
     */
    public static NemesisBlockHeaderBuilder create(SignatureDto signature, KeyDto signerPublicKey, byte version, NetworkTypeDto network, EntityTypeDto type, HeightDto height, TimestampDto timestamp, DifficultyDto difficulty, VrfProofBuilder generationHashProof, Hash256Dto previousBlockHash, Hash256Dto transactionsHash, Hash256Dto receiptsHash, Hash256Dto stateHash, AddressDto beneficiaryAddress, BlockFeeMultiplierDto feeMultiplier, int votingEligibleAccountsCount, long harvestingEligibleAccountsCount, AmountDto totalVotingBalance, Hash256Dto previousImportanceBlockHash) {
        return new NemesisBlockHeaderBuilder(signature, signerPublicKey, version, network, type, height, timestamp, difficulty, generationHashProof, previousBlockHash, transactionsHash, receiptsHash, stateHash, beneficiaryAddress, feeMultiplier, votingEligibleAccountsCount, harvestingEligibleAccountsCount, totalVotingBalance, previousImportanceBlockHash);
    }

    /**
     * Gets number of voting eligible accounts.
     *
     * @return Number of voting eligible accounts.
     */
    public int getVotingEligibleAccountsCount() {
        return this.importanceBlockFooter.getVotingEligibleAccountsCount();
    }

    /**
     * Gets number of harvesting eligible accounts.
     *
     * @return Number of harvesting eligible accounts.
     */
    public long getHarvestingEligibleAccountsCount() {
        return this.importanceBlockFooter.getHarvestingEligibleAccountsCount();
    }

    /**
     * Gets total balance eligible for voting.
     *
     * @return Total balance eligible for voting.
     */
    public AmountDto getTotalVotingBalance() {
        return this.importanceBlockFooter.getTotalVotingBalance();
    }

    /**
     * Gets previous importance block hash.
     *
     * @return Previous importance block hash.
     */
    public Hash256Dto getPreviousImportanceBlockHash() {
        return this.importanceBlockFooter.getPreviousImportanceBlockHash();
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += this.importanceBlockFooter.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            GeneratorUtils.writeEntity(dataOutputStream, this.importanceBlockFooter);
        });
    }
}

