package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a namespace expiry receipt
**/
public class NamespaceExpiryReceiptBuilder extends ReceiptBuilder implements Serializer {

    /** Expiring namespace id. **/
    private final NamespaceIdDto artifactId;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected NamespaceExpiryReceiptBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.artifactId = NamespaceIdDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of NamespaceExpiryReceiptBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of NamespaceExpiryReceiptBuilder.
     */
    public static NamespaceExpiryReceiptBuilder loadFromBinary(DataInputStream stream) {
        return new NamespaceExpiryReceiptBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param version Receipt version.
    * @param type Receipt type.
    * @param artifactId Expiring namespace id.
    */
    protected NamespaceExpiryReceiptBuilder(short version, ReceiptTypeDto type, NamespaceIdDto artifactId) {
        super(version, type);
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(artifactId, "artifactId is null");
        this.artifactId = artifactId;
    }
    
    /**
     * Creates an instance of NamespaceExpiryReceiptBuilder.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param artifactId Expiring namespace id.
     * @return Instance of NamespaceExpiryReceiptBuilder.
     */
    public static NamespaceExpiryReceiptBuilder create(short version, ReceiptTypeDto type, NamespaceIdDto artifactId) {
        return new NamespaceExpiryReceiptBuilder(version, type, artifactId);
    }

    /**
     * Gets expiring namespace id.
     *
     * @return Expiring namespace id.
     */
    public NamespaceIdDto getArtifactId() {
        return this.artifactId;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += this.artifactId.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            GeneratorUtils.writeEntity(dataOutputStream, this.artifactId);
        });
    }
}

