package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a mosaic supply revocation transaction
**/
public class MosaicSupplyRevocationTransactionBodyBuilder implements Serializer {

    /** Address from which tokens should be revoked. **/
    private final UnresolvedAddressDto sourceAddress;

    /** Revoked mosaic and amount. **/
    private final UnresolvedMosaicBuilder mosaic;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MosaicSupplyRevocationTransactionBodyBuilder(DataInputStream stream) {
        try {
            this.sourceAddress = UnresolvedAddressDto.loadFromBinary(stream);
            this.mosaic = UnresolvedMosaicBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of MosaicSupplyRevocationTransactionBodyBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicSupplyRevocationTransactionBodyBuilder.
     */
    public static MosaicSupplyRevocationTransactionBodyBuilder loadFromBinary(DataInputStream stream) {
        return new MosaicSupplyRevocationTransactionBodyBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param sourceAddress Address from which tokens should be revoked.
    * @param mosaic Revoked mosaic and amount.
    */
    protected MosaicSupplyRevocationTransactionBodyBuilder(UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        GeneratorUtils.notNull(sourceAddress, "sourceAddress is null");
        GeneratorUtils.notNull(mosaic, "mosaic is null");
        this.sourceAddress = sourceAddress;
        this.mosaic = mosaic;
    }
    
    /**
     * Creates an instance of MosaicSupplyRevocationTransactionBodyBuilder.
     *
     * @param sourceAddress Address from which tokens should be revoked.
     * @param mosaic Revoked mosaic and amount.
     * @return Instance of MosaicSupplyRevocationTransactionBodyBuilder.
     */
    public static MosaicSupplyRevocationTransactionBodyBuilder create(UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        return new MosaicSupplyRevocationTransactionBodyBuilder(sourceAddress, mosaic);
    }

    /**
     * Gets address from which tokens should be revoked.
     *
     * @return Address from which tokens should be revoked.
     */
    public UnresolvedAddressDto getSourceAddress() {
        return this.sourceAddress;
    }

    /**
     * Gets revoked mosaic and amount.
     *
     * @return Revoked mosaic and amount.
     */
    public UnresolvedMosaicBuilder getMosaic() {
        return this.mosaic;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.sourceAddress.getSize();
        size += this.mosaic.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.sourceAddress);
            GeneratorUtils.writeEntity(dataOutputStream, this.mosaic);
        });
    }
}

