package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for an embedded voting key link transaction
**/
public class EmbeddedVotingKeyLinkTransactionBuilder extends EmbeddedTransactionBuilder implements Serializer {

    /** Voting key link transaction body. **/
    private final VotingKeyLinkTransactionBodyBuilder votingKeyLinkTransactionBody;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected EmbeddedVotingKeyLinkTransactionBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.votingKeyLinkTransactionBody = VotingKeyLinkTransactionBodyBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of EmbeddedVotingKeyLinkTransactionBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of EmbeddedVotingKeyLinkTransactionBuilder.
     */
    public static EmbeddedVotingKeyLinkTransactionBuilder loadFromBinary(DataInputStream stream) {
        return new EmbeddedVotingKeyLinkTransactionBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param signerPublicKey Entity signer's public key.
    * @param version Entity version.
    * @param network Entity network.
    * @param type Entity type.
    * @param linkedPublicKey Linked public key.
    * @param startEpoch Start finalization epoch.
    * @param endEpoch End finalization epoch.
    * @param linkAction Link action.
    */
    protected EmbeddedVotingKeyLinkTransactionBuilder(KeyDto signerPublicKey, byte version, NetworkTypeDto network, EntityTypeDto type, VotingKeyDto linkedPublicKey, FinalizationEpochDto startEpoch, FinalizationEpochDto endEpoch, LinkActionDto linkAction) {
        super(signerPublicKey, version, network, type);
        GeneratorUtils.notNull(signerPublicKey, "signerPublicKey is null");
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(network, "network is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(linkedPublicKey, "linkedPublicKey is null");
        GeneratorUtils.notNull(startEpoch, "startEpoch is null");
        GeneratorUtils.notNull(endEpoch, "endEpoch is null");
        GeneratorUtils.notNull(linkAction, "linkAction is null");
        this.votingKeyLinkTransactionBody = new VotingKeyLinkTransactionBodyBuilder(linkedPublicKey, startEpoch, endEpoch, linkAction);
    }
    
    /**
     * Creates an instance of EmbeddedVotingKeyLinkTransactionBuilder.
     *
     * @param signerPublicKey Entity signer's public key.
     * @param version Entity version.
     * @param network Entity network.
     * @param type Entity type.
     * @param linkedPublicKey Linked public key.
     * @param startEpoch Start finalization epoch.
     * @param endEpoch End finalization epoch.
     * @param linkAction Link action.
     * @return Instance of EmbeddedVotingKeyLinkTransactionBuilder.
     */
    public static EmbeddedVotingKeyLinkTransactionBuilder create(KeyDto signerPublicKey, byte version, NetworkTypeDto network, EntityTypeDto type, VotingKeyDto linkedPublicKey, FinalizationEpochDto startEpoch, FinalizationEpochDto endEpoch, LinkActionDto linkAction) {
        return new EmbeddedVotingKeyLinkTransactionBuilder(signerPublicKey, version, network, type, linkedPublicKey, startEpoch, endEpoch, linkAction);
    }

    /**
     * Gets linked public key.
     *
     * @return Linked public key.
     */
    public VotingKeyDto getLinkedPublicKey() {
        return this.votingKeyLinkTransactionBody.getLinkedPublicKey();
    }

    /**
     * Gets start finalization epoch.
     *
     * @return Start finalization epoch.
     */
    public FinalizationEpochDto getStartEpoch() {
        return this.votingKeyLinkTransactionBody.getStartEpoch();
    }

    /**
     * Gets end finalization epoch.
     *
     * @return End finalization epoch.
     */
    public FinalizationEpochDto getEndEpoch() {
        return this.votingKeyLinkTransactionBody.getEndEpoch();
    }

    /**
     * Gets link action.
     *
     * @return Link action.
     */
    public LinkActionDto getLinkAction() {
        return this.votingKeyLinkTransactionBody.getLinkAction();
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += this.votingKeyLinkTransactionBody.getSize();
        return size;
    }

    /**
     * Gets the body builder of the object.
     *
     * @return Body builder.
     */
    @Override
    public VotingKeyLinkTransactionBodyBuilder getBody() {
        return this.votingKeyLinkTransactionBody;
    }


    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            GeneratorUtils.writeEntity(dataOutputStream, this.votingKeyLinkTransactionBody);
        });
    }
}

