package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for an embedded mosaic supply revocation transaction
**/
public class EmbeddedMosaicSupplyRevocationTransactionBuilder extends EmbeddedTransactionBuilder implements Serializer {

    /** Mosaic supply revocation transaction body. **/
    private final MosaicSupplyRevocationTransactionBodyBuilder mosaicSupplyRevocationTransactionBody;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected EmbeddedMosaicSupplyRevocationTransactionBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.mosaicSupplyRevocationTransactionBody = MosaicSupplyRevocationTransactionBodyBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of EmbeddedMosaicSupplyRevocationTransactionBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of EmbeddedMosaicSupplyRevocationTransactionBuilder.
     */
    public static EmbeddedMosaicSupplyRevocationTransactionBuilder loadFromBinary(DataInputStream stream) {
        return new EmbeddedMosaicSupplyRevocationTransactionBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param signerPublicKey Entity signer's public key.
    * @param version Entity version.
    * @param network Entity network.
    * @param type Entity type.
    * @param sourceAddress Address from which tokens should be revoked.
    * @param mosaic Revoked mosaic and amount.
    */
    protected EmbeddedMosaicSupplyRevocationTransactionBuilder(KeyDto signerPublicKey, byte version, NetworkTypeDto network, EntityTypeDto type, UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        super(signerPublicKey, version, network, type);
        GeneratorUtils.notNull(signerPublicKey, "signerPublicKey is null");
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(network, "network is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(sourceAddress, "sourceAddress is null");
        GeneratorUtils.notNull(mosaic, "mosaic is null");
        this.mosaicSupplyRevocationTransactionBody = new MosaicSupplyRevocationTransactionBodyBuilder(sourceAddress, mosaic);
    }
    
    /**
     * Creates an instance of EmbeddedMosaicSupplyRevocationTransactionBuilder.
     *
     * @param signerPublicKey Entity signer's public key.
     * @param version Entity version.
     * @param network Entity network.
     * @param type Entity type.
     * @param sourceAddress Address from which tokens should be revoked.
     * @param mosaic Revoked mosaic and amount.
     * @return Instance of EmbeddedMosaicSupplyRevocationTransactionBuilder.
     */
    public static EmbeddedMosaicSupplyRevocationTransactionBuilder create(KeyDto signerPublicKey, byte version, NetworkTypeDto network, EntityTypeDto type, UnresolvedAddressDto sourceAddress, UnresolvedMosaicBuilder mosaic) {
        return new EmbeddedMosaicSupplyRevocationTransactionBuilder(signerPublicKey, version, network, type, sourceAddress, mosaic);
    }

    /**
     * Gets address from which tokens should be revoked.
     *
     * @return Address from which tokens should be revoked.
     */
    public UnresolvedAddressDto getSourceAddress() {
        return this.mosaicSupplyRevocationTransactionBody.getSourceAddress();
    }

    /**
     * Gets revoked mosaic and amount.
     *
     * @return Revoked mosaic and amount.
     */
    public UnresolvedMosaicBuilder getMosaic() {
        return this.mosaicSupplyRevocationTransactionBody.getMosaic();
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += this.mosaicSupplyRevocationTransactionBody.getSize();
        return size;
    }

    /**
     * Gets the body builder of the object.
     *
     * @return Body builder.
     */
    @Override
    public MosaicSupplyRevocationTransactionBodyBuilder getBody() {
        return this.mosaicSupplyRevocationTransactionBody;
    }


    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            GeneratorUtils.writeEntity(dataOutputStream, this.mosaicSupplyRevocationTransactionBody);
        });
    }
}

