package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Cosignature attached to an aggregate transaction
**/
public class CosignatureBuilder implements Serializer {

    /** Version. **/
    private final long version;

    /** Cosigner public key. **/
    private final KeyDto signerPublicKey;

    /** Cosigner signature. **/
    private final SignatureDto signature;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected CosignatureBuilder(DataInputStream stream) {
        try {
            this.version = Long.reverseBytes(stream.readLong());
            this.signerPublicKey = KeyDto.loadFromBinary(stream);
            this.signature = SignatureDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of CosignatureBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of CosignatureBuilder.
     */
    public static CosignatureBuilder loadFromBinary(DataInputStream stream) {
        return new CosignatureBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param version Version.
    * @param signerPublicKey Cosigner public key.
    * @param signature Cosigner signature.
    */
    protected CosignatureBuilder(long version, KeyDto signerPublicKey, SignatureDto signature) {
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(signerPublicKey, "signerPublicKey is null");
        GeneratorUtils.notNull(signature, "signature is null");
        this.version = version;
        this.signerPublicKey = signerPublicKey;
        this.signature = signature;
    }
    
    /**
     * Creates an instance of CosignatureBuilder.
     *
     * @param version Version.
     * @param signerPublicKey Cosigner public key.
     * @param signature Cosigner signature.
     * @return Instance of CosignatureBuilder.
     */
    public static CosignatureBuilder create(long version, KeyDto signerPublicKey, SignatureDto signature) {
        return new CosignatureBuilder(version, signerPublicKey, signature);
    }

    /**
     * Gets version.
     *
     * @return Version.
     */
    public long getVersion() {
        return this.version;
    }

    /**
     * Gets cosigner public key.
     *
     * @return Cosigner public key.
     */
    public KeyDto getSignerPublicKey() {
        return this.signerPublicKey;
    }

    /**
     * Gets cosigner signature.
     *
     * @return Cosigner signature.
     */
    public SignatureDto getSignature() {
        return this.signature;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += 8; // version
        size += this.signerPublicKey.getSize();
        size += this.signature.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            dataOutputStream.writeLong(Long.reverseBytes((long) this.getVersion()));
            GeneratorUtils.writeEntity(dataOutputStream, this.signerPublicKey);
            GeneratorUtils.writeEntity(dataOutputStream, this.signature);
        });
    }
}

