package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;

/**
* Enumeration of account types
**/
public enum AccountTypeDto implements Serializer {

    /** account is not linked to another account. */
    UNLINKED((byte) 0),

    /** account is a balance-holding account that is linked to a remote harvester account. */
    MAIN((byte) 1),

    /** account is a remote harvester account that is linked to a balance-holding account. */
    REMOTE((byte) 2),

    /** account is a remote harvester eligible account that is unlinked \note this allows an account that has previously been used as remote to be reused as a remote. */
    REMOTE_UNLINKED((byte) 3);


    /** Enum value. */
    private final byte value;

    /**
     * Constructor.
     *
     * @param value Enum value.
     */
     AccountTypeDto(final byte value) {
        this.value = value;
    }

    /**
     * Gets enum value.
     *
     * @param value Raw value of the enum.
     * @return Enum value.
     */
    public static AccountTypeDto rawValueOf(final byte value) {
        for (AccountTypeDto current : AccountTypeDto.values()) {
            if (value == current.value) {
                return current;
            }
        }
        throw new IllegalArgumentException(value + " was not a backing value for AccountTypeDto.");
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        return 1;
    }

    /**
     * Gets the value of the enum.
     *
     * @return Value of the enum.
     */
    public byte getValue() {
        return this.value;
    }
    /**
     * Creates an instance of AccountTypeDto from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of AccountTypeDto.
     */
    public static AccountTypeDto loadFromBinary(final DataInputStream stream) {
        try {
            final byte streamValue = stream.readByte();
            return rawValueOf(streamValue);
        } catch(Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            dataOutputStream.writeByte(this.value);
        });
    }
}

