package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a namespace path
**/
public class NamespacePathBuilder implements Serializer {

    /** Namespace path (excluding root id). **/
    private final List<NamespaceIdDto> path;

    /** Namespace alias. **/
    private final NamespaceAliasBuilder alias;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected NamespacePathBuilder(DataInputStream stream) {
        try {
            final byte pathSize = stream.readByte();
            this.path = GeneratorUtils.loadFromBinaryArray(NamespaceIdDto::loadFromBinary, stream, pathSize);
            this.alias = NamespaceAliasBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of NamespacePathBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of NamespacePathBuilder.
     */
    public static NamespacePathBuilder loadFromBinary(DataInputStream stream) {
        return new NamespacePathBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param path Namespace path (excluding root id).
    * @param alias Namespace alias.
    */
    protected NamespacePathBuilder(List<NamespaceIdDto> path, NamespaceAliasBuilder alias) {
        GeneratorUtils.notNull(path, "path is null");
        GeneratorUtils.notNull(alias, "alias is null");
        this.path = path;
        this.alias = alias;
    }
    
    /**
     * Creates an instance of NamespacePathBuilder.
     *
     * @param path Namespace path (excluding root id).
     * @param alias Namespace alias.
     * @return Instance of NamespacePathBuilder.
     */
    public static NamespacePathBuilder create(List<NamespaceIdDto> path, NamespaceAliasBuilder alias) {
        return new NamespacePathBuilder(path, alias);
    }

    /**
     * Gets namespace path (excluding root id).
     *
     * @return Namespace path (excluding root id).
     */
    public List<NamespaceIdDto> getPath() {
        return this.path;
    }

    /**
     * Gets namespace alias.
     *
     * @return Namespace alias.
     */
    public NamespaceAliasBuilder getAlias() {
        return this.alias;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += 1; // pathSize
        size += this.path.stream().mapToInt(o -> o.getSize()).sum();
        size += this.alias.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            dataOutputStream.writeByte((byte) GeneratorUtils.getSize(this.getPath()));
            GeneratorUtils.writeList(dataOutputStream, this.path);
            GeneratorUtils.writeEntity(dataOutputStream, this.alias);
        });
    }
}

