package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;

/** Mosaic nonce. */
public final class MosaicNonceDto implements Serializer {
    /** Mosaic nonce. */
    private final int mosaicNonce;

    /**
     * Constructor.
     *
     * @param mosaicNonce Mosaic nonce.
     */
    public MosaicNonceDto(final int mosaicNonce) {
        this.mosaicNonce = mosaicNonce;
    }

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize.
     */
    public MosaicNonceDto(final DataInputStream stream) {
        try {
            this.mosaicNonce = Integer.reverseBytes(stream.readInt());
        } catch(Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Gets Mosaic nonce.
     *
     * @return Mosaic nonce.
     */
    public int getMosaicNonce() {
        return this.mosaicNonce;
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        return 4;
    }

    /**
     * Creates an instance of MosaicNonceDto from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicNonceDto.
     */
    public static MosaicNonceDto loadFromBinary(final DataInputStream stream) {
        return new MosaicNonceDto(stream);
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            dataOutputStream.writeInt(Integer.reverseBytes(this.getMosaicNonce()));
        });
    }
}

