package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a mosaic expiry receipt
**/
public class MosaicExpiryReceiptBuilder extends ReceiptBuilder implements Serializer {

    /** Expiring mosaic id. **/
    private final MosaicIdDto artifactId;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MosaicExpiryReceiptBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.artifactId = MosaicIdDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of MosaicExpiryReceiptBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicExpiryReceiptBuilder.
     */
    public static MosaicExpiryReceiptBuilder loadFromBinary(DataInputStream stream) {
        return new MosaicExpiryReceiptBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param version Receipt version.
    * @param type Receipt type.
    * @param artifactId Expiring mosaic id.
    */
    protected MosaicExpiryReceiptBuilder(short version, ReceiptTypeDto type, MosaicIdDto artifactId) {
        super(version, type);
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(artifactId, "artifactId is null");
        this.artifactId = artifactId;
    }
    
    /**
     * Creates an instance of MosaicExpiryReceiptBuilder.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param artifactId Expiring mosaic id.
     * @return Instance of MosaicExpiryReceiptBuilder.
     */
    public static MosaicExpiryReceiptBuilder create(short version, ReceiptTypeDto type, MosaicIdDto artifactId) {
        return new MosaicExpiryReceiptBuilder(version, type, artifactId);
    }

    /**
     * Gets expiring mosaic id.
     *
     * @return Expiring mosaic id.
     */
    public MosaicIdDto getArtifactId() {
        return this.artifactId;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += this.artifactId.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            GeneratorUtils.writeEntity(dataOutputStream, this.artifactId);
        });
    }
}

