package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for an mosaic alias transaction
**/
public class MosaicAliasTransactionBodyBuilder implements Serializer {

    /** Identifier of the namespace that will become an alias. **/
    private final NamespaceIdDto namespaceId;

    /** Aliased mosaic identifier. **/
    private final MosaicIdDto mosaicId;

    /** Alias action. **/
    private final AliasActionDto aliasAction;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MosaicAliasTransactionBodyBuilder(DataInputStream stream) {
        try {
            this.namespaceId = NamespaceIdDto.loadFromBinary(stream);
            this.mosaicId = MosaicIdDto.loadFromBinary(stream);
            this.aliasAction = AliasActionDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of MosaicAliasTransactionBodyBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicAliasTransactionBodyBuilder.
     */
    public static MosaicAliasTransactionBodyBuilder loadFromBinary(DataInputStream stream) {
        return new MosaicAliasTransactionBodyBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param namespaceId Identifier of the namespace that will become an alias.
    * @param mosaicId Aliased mosaic identifier.
    * @param aliasAction Alias action.
    */
    protected MosaicAliasTransactionBodyBuilder(NamespaceIdDto namespaceId, MosaicIdDto mosaicId, AliasActionDto aliasAction) {
        GeneratorUtils.notNull(namespaceId, "namespaceId is null");
        GeneratorUtils.notNull(mosaicId, "mosaicId is null");
        GeneratorUtils.notNull(aliasAction, "aliasAction is null");
        this.namespaceId = namespaceId;
        this.mosaicId = mosaicId;
        this.aliasAction = aliasAction;
    }
    
    /**
     * Creates an instance of MosaicAliasTransactionBodyBuilder.
     *
     * @param namespaceId Identifier of the namespace that will become an alias.
     * @param mosaicId Aliased mosaic identifier.
     * @param aliasAction Alias action.
     * @return Instance of MosaicAliasTransactionBodyBuilder.
     */
    public static MosaicAliasTransactionBodyBuilder create(NamespaceIdDto namespaceId, MosaicIdDto mosaicId, AliasActionDto aliasAction) {
        return new MosaicAliasTransactionBodyBuilder(namespaceId, mosaicId, aliasAction);
    }

    /**
     * Gets identifier of the namespace that will become an alias.
     *
     * @return Identifier of the namespace that will become an alias.
     */
    public NamespaceIdDto getNamespaceId() {
        return this.namespaceId;
    }

    /**
     * Gets aliased mosaic identifier.
     *
     * @return Aliased mosaic identifier.
     */
    public MosaicIdDto getMosaicId() {
        return this.mosaicId;
    }

    /**
     * Gets alias action.
     *
     * @return Alias action.
     */
    public AliasActionDto getAliasAction() {
        return this.aliasAction;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.namespaceId.getSize();
        size += this.mosaicId.getSize();
        size += this.aliasAction.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.namespaceId);
            GeneratorUtils.writeEntity(dataOutputStream, this.mosaicId);
            GeneratorUtils.writeEntity(dataOutputStream, this.aliasAction);
        });
    }
}

