package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a global restriction key-value set
**/
public class GlobalKeyValueSetBuilder implements Serializer {

    /** Key value array. **/
    private final List<GlobalKeyValueBuilder> keys;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected GlobalKeyValueSetBuilder(DataInputStream stream) {
        try {
            final byte keyValueCount = stream.readByte();
            this.keys = GeneratorUtils.loadFromBinaryArray(GlobalKeyValueBuilder::loadFromBinary, stream, keyValueCount);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of GlobalKeyValueSetBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of GlobalKeyValueSetBuilder.
     */
    public static GlobalKeyValueSetBuilder loadFromBinary(DataInputStream stream) {
        return new GlobalKeyValueSetBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param keys Key value array.
    */
    protected GlobalKeyValueSetBuilder(List<GlobalKeyValueBuilder> keys) {
        GeneratorUtils.notNull(keys, "keys is null");
        this.keys = keys;
    }
    
    /**
     * Creates an instance of GlobalKeyValueSetBuilder.
     *
     * @param keys Key value array.
     * @return Instance of GlobalKeyValueSetBuilder.
     */
    public static GlobalKeyValueSetBuilder create(List<GlobalKeyValueBuilder> keys) {
        return new GlobalKeyValueSetBuilder(keys);
    }

    /**
     * Gets key value array.
     *
     * @return Key value array.
     */
    public List<GlobalKeyValueBuilder> getKeys() {
        return this.keys;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += 1; // keyValueCount
        size += this.keys.stream().mapToInt(o -> o.getSize()).sum();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            dataOutputStream.writeByte((byte) GeneratorUtils.getSize(this.getKeys()));
            GeneratorUtils.writeList(dataOutputStream, this.keys);
        });
    }
}

