package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a global key-value
**/
public class GlobalKeyValueBuilder implements Serializer {

    /** Key associated with a restriction rule. **/
    private final MosaicRestrictionKeyDto key;

    /** Restriction rule (the value) associated with a key. **/
    private final RestrictionRuleBuilder restrictionRule;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected GlobalKeyValueBuilder(DataInputStream stream) {
        try {
            this.key = MosaicRestrictionKeyDto.loadFromBinary(stream);
            this.restrictionRule = RestrictionRuleBuilder.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of GlobalKeyValueBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of GlobalKeyValueBuilder.
     */
    public static GlobalKeyValueBuilder loadFromBinary(DataInputStream stream) {
        return new GlobalKeyValueBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param key Key associated with a restriction rule.
    * @param restrictionRule Restriction rule (the value) associated with a key.
    */
    protected GlobalKeyValueBuilder(MosaicRestrictionKeyDto key, RestrictionRuleBuilder restrictionRule) {
        GeneratorUtils.notNull(key, "key is null");
        GeneratorUtils.notNull(restrictionRule, "restrictionRule is null");
        this.key = key;
        this.restrictionRule = restrictionRule;
    }
    
    /**
     * Creates an instance of GlobalKeyValueBuilder.
     *
     * @param key Key associated with a restriction rule.
     * @param restrictionRule Restriction rule (the value) associated with a key.
     * @return Instance of GlobalKeyValueBuilder.
     */
    public static GlobalKeyValueBuilder create(MosaicRestrictionKeyDto key, RestrictionRuleBuilder restrictionRule) {
        return new GlobalKeyValueBuilder(key, restrictionRule);
    }

    /**
     * Gets key associated with a restriction rule.
     *
     * @return Key associated with a restriction rule.
     */
    public MosaicRestrictionKeyDto getKey() {
        return this.key;
    }

    /**
     * Gets restriction rule (the value) associated with a key.
     *
     * @return Restriction rule (the value) associated with a key.
     */
    public RestrictionRuleBuilder getRestrictionRule() {
        return this.restrictionRule;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.key.getSize();
        size += this.restrictionRule.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.key);
            GeneratorUtils.writeEntity(dataOutputStream, this.restrictionRule);
        });
    }
}

