package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for account restrictions
**/
public class AccountRestrictionsBuilder extends StateHeaderBuilder implements Serializer {

    /** Address on which restrictions are placed. **/
    private final AddressDto address;

    /** Account restrictions. **/
    private final List<AccountRestrictionsInfoBuilder> restrictions;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected AccountRestrictionsBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.address = AddressDto.loadFromBinary(stream);
            final long restrictionsCount = Long.reverseBytes(stream.readLong());
            this.restrictions = GeneratorUtils.loadFromBinaryArray(AccountRestrictionsInfoBuilder::loadFromBinary, stream, restrictionsCount);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of AccountRestrictionsBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of AccountRestrictionsBuilder.
     */
    public static AccountRestrictionsBuilder loadFromBinary(DataInputStream stream) {
        return new AccountRestrictionsBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param version Serialization version.
    * @param address Address on which restrictions are placed.
    * @param restrictions Account restrictions.
    */
    protected AccountRestrictionsBuilder(short version, AddressDto address, List<AccountRestrictionsInfoBuilder> restrictions) {
        super(version);
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(address, "address is null");
        GeneratorUtils.notNull(restrictions, "restrictions is null");
        this.address = address;
        this.restrictions = restrictions;
    }
    
    /**
     * Creates an instance of AccountRestrictionsBuilder.
     *
     * @param version Serialization version.
     * @param address Address on which restrictions are placed.
     * @param restrictions Account restrictions.
     * @return Instance of AccountRestrictionsBuilder.
     */
    public static AccountRestrictionsBuilder create(short version, AddressDto address, List<AccountRestrictionsInfoBuilder> restrictions) {
        return new AccountRestrictionsBuilder(version, address, restrictions);
    }

    /**
     * Gets address on which restrictions are placed.
     *
     * @return Address on which restrictions are placed.
     */
    public AddressDto getAddress() {
        return this.address;
    }

    /**
     * Gets account restrictions.
     *
     * @return Account restrictions.
     */
    public List<AccountRestrictionsInfoBuilder> getRestrictions() {
        return this.restrictions;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += this.address.getSize();
        size += 8; // restrictionsCount
        size += this.restrictions.stream().mapToInt(o -> o.getSize()).sum();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            GeneratorUtils.writeEntity(dataOutputStream, this.address);
            dataOutputStream.writeLong(Long.reverseBytes((long) GeneratorUtils.getSize(this.getRestrictions())));
            GeneratorUtils.writeList(dataOutputStream, this.restrictions);
        });
    }
}

