package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for transaction type based account restriction
**/
public class AccountRestrictionTransactionTypeValueBuilder implements Serializer {

    /** Restriction values. **/
    private final List<EntityTypeDto> restrictionValues;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected AccountRestrictionTransactionTypeValueBuilder(DataInputStream stream) {
        try {
            final long restrictionValuesCount = Long.reverseBytes(stream.readLong());
            this.restrictionValues = GeneratorUtils.loadFromBinaryArray(EntityTypeDto::loadFromBinary, stream, restrictionValuesCount);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of AccountRestrictionTransactionTypeValueBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of AccountRestrictionTransactionTypeValueBuilder.
     */
    public static AccountRestrictionTransactionTypeValueBuilder loadFromBinary(DataInputStream stream) {
        return new AccountRestrictionTransactionTypeValueBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param restrictionValues Restriction values.
    */
    protected AccountRestrictionTransactionTypeValueBuilder(List<EntityTypeDto> restrictionValues) {
        GeneratorUtils.notNull(restrictionValues, "restrictionValues is null");
        this.restrictionValues = restrictionValues;
    }
    
    /**
     * Creates an instance of AccountRestrictionTransactionTypeValueBuilder.
     *
     * @param restrictionValues Restriction values.
     * @return Instance of AccountRestrictionTransactionTypeValueBuilder.
     */
    public static AccountRestrictionTransactionTypeValueBuilder create(List<EntityTypeDto> restrictionValues) {
        return new AccountRestrictionTransactionTypeValueBuilder(restrictionValues);
    }

    /**
     * Gets restriction values.
     *
     * @return Restriction values.
     */
    public List<EntityTypeDto> getRestrictionValues() {
        return this.restrictionValues;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += 8; // restrictionValuesCount
        size += this.restrictionValues.stream().mapToInt(o -> o.getSize()).sum();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            dataOutputStream.writeLong(Long.reverseBytes((long) GeneratorUtils.getSize(this.getRestrictionValues())));
            GeneratorUtils.writeList(dataOutputStream, this.restrictionValues);
        });
    }
}

