package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for address based account restriction
**/
public class AccountRestrictionAddressValueBuilder implements Serializer {

    /** Restriction values. **/
    private final List<AddressDto> restrictionValues;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected AccountRestrictionAddressValueBuilder(DataInputStream stream) {
        try {
            final long restrictionValuesCount = Long.reverseBytes(stream.readLong());
            this.restrictionValues = GeneratorUtils.loadFromBinaryArray(AddressDto::loadFromBinary, stream, restrictionValuesCount);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of AccountRestrictionAddressValueBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of AccountRestrictionAddressValueBuilder.
     */
    public static AccountRestrictionAddressValueBuilder loadFromBinary(DataInputStream stream) {
        return new AccountRestrictionAddressValueBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param restrictionValues Restriction values.
    */
    protected AccountRestrictionAddressValueBuilder(List<AddressDto> restrictionValues) {
        GeneratorUtils.notNull(restrictionValues, "restrictionValues is null");
        this.restrictionValues = restrictionValues;
    }
    
    /**
     * Creates an instance of AccountRestrictionAddressValueBuilder.
     *
     * @param restrictionValues Restriction values.
     * @return Instance of AccountRestrictionAddressValueBuilder.
     */
    public static AccountRestrictionAddressValueBuilder create(List<AddressDto> restrictionValues) {
        return new AccountRestrictionAddressValueBuilder(restrictionValues);
    }

    /**
     * Gets restriction values.
     *
     * @return Restriction values.
     */
    public List<AddressDto> getRestrictionValues() {
        return this.restrictionValues;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += 8; // restrictionValuesCount
        size += this.restrictionValues.stream().mapToInt(o -> o.getSize()).sum();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            dataOutputStream.writeLong(Long.reverseBytes((long) GeneratorUtils.getSize(this.getRestrictionValues())));
            GeneratorUtils.writeList(dataOutputStream, this.restrictionValues);
        });
    }
}

