/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a normal block header
**/
public class NormalBlockHeaderBuilder extends BlockHeaderBuilder implements Serializer {

    /** Reserved padding to align end of BlockHeader on 8-byte boundary. **/
    private final int blockHeader_Reserved1;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected NormalBlockHeaderBuilder(DataInputStream stream) {
        super(stream);
        try {
            this.blockHeader_Reserved1 = Integer.reverseBytes(stream.readInt());
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of NormalBlockHeaderBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of NormalBlockHeaderBuilder.
     */
    public static NormalBlockHeaderBuilder loadFromBinary(DataInputStream stream) {
        return new NormalBlockHeaderBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param signature Entity signature.
    * @param signerPublicKey Entity signer's public key.
    * @param version Entity version.
    * @param network Entity network.
    * @param type Entity type.
    * @param height Block height.
    * @param timestamp Number of milliseconds elapsed since creation of nemesis block.
    * @param difficulty Block difficulty.
    * @param generationHashProof Generation hash proof.
    * @param previousBlockHash Previous block hash.
    * @param transactionsHash Hash of the transactions in this block.
    * @param receiptsHash Hash of the receipts generated by this block.
    * @param stateHash Hash of the global chain state at this block.
    * @param beneficiaryAddress Beneficiary address designated by harvester.
    * @param feeMultiplier Fee multiplier applied to block transactions.
    */
    protected NormalBlockHeaderBuilder(SignatureDto signature, KeyDto signerPublicKey, byte version, NetworkTypeDto network, EntityTypeDto type, HeightDto height, TimestampDto timestamp, DifficultyDto difficulty, VrfProofBuilder generationHashProof, Hash256Dto previousBlockHash, Hash256Dto transactionsHash, Hash256Dto receiptsHash, Hash256Dto stateHash, AddressDto beneficiaryAddress, BlockFeeMultiplierDto feeMultiplier) {
        super(signature, signerPublicKey, version, network, type, height, timestamp, difficulty, generationHashProof, previousBlockHash, transactionsHash, receiptsHash, stateHash, beneficiaryAddress, feeMultiplier);
        GeneratorUtils.notNull(signature, "signature is null");
        GeneratorUtils.notNull(signerPublicKey, "signerPublicKey is null");
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(network, "network is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(height, "height is null");
        GeneratorUtils.notNull(timestamp, "timestamp is null");
        GeneratorUtils.notNull(difficulty, "difficulty is null");
        GeneratorUtils.notNull(generationHashProof, "generationHashProof is null");
        GeneratorUtils.notNull(previousBlockHash, "previousBlockHash is null");
        GeneratorUtils.notNull(transactionsHash, "transactionsHash is null");
        GeneratorUtils.notNull(receiptsHash, "receiptsHash is null");
        GeneratorUtils.notNull(stateHash, "stateHash is null");
        GeneratorUtils.notNull(beneficiaryAddress, "beneficiaryAddress is null");
        GeneratorUtils.notNull(feeMultiplier, "feeMultiplier is null");
        this.blockHeader_Reserved1 = 0;
    }
    
    /**
     * Creates an instance of NormalBlockHeaderBuilder.
     *
     * @param signature Entity signature.
     * @param signerPublicKey Entity signer's public key.
     * @param version Entity version.
     * @param network Entity network.
     * @param type Entity type.
     * @param height Block height.
     * @param timestamp Number of milliseconds elapsed since creation of nemesis block.
     * @param difficulty Block difficulty.
     * @param generationHashProof Generation hash proof.
     * @param previousBlockHash Previous block hash.
     * @param transactionsHash Hash of the transactions in this block.
     * @param receiptsHash Hash of the receipts generated by this block.
     * @param stateHash Hash of the global chain state at this block.
     * @param beneficiaryAddress Beneficiary address designated by harvester.
     * @param feeMultiplier Fee multiplier applied to block transactions.
     * @return Instance of NormalBlockHeaderBuilder.
     */
    public static NormalBlockHeaderBuilder create(SignatureDto signature, KeyDto signerPublicKey, byte version, NetworkTypeDto network, EntityTypeDto type, HeightDto height, TimestampDto timestamp, DifficultyDto difficulty, VrfProofBuilder generationHashProof, Hash256Dto previousBlockHash, Hash256Dto transactionsHash, Hash256Dto receiptsHash, Hash256Dto stateHash, AddressDto beneficiaryAddress, BlockFeeMultiplierDto feeMultiplier) {
        return new NormalBlockHeaderBuilder(signature, signerPublicKey, version, network, type, height, timestamp, difficulty, generationHashProof, previousBlockHash, transactionsHash, receiptsHash, stateHash, beneficiaryAddress, feeMultiplier);
    }

    /**
     * Gets reserved padding to align end of BlockHeader on 8-byte boundary.
     *
     * @return Reserved padding to align end of BlockHeader on 8-byte boundary.
     */
    private int getBlockHeader_Reserved1() {
        return this.blockHeader_Reserved1;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = super.getSize();
        size += 4; // blockHeader_Reserved1
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            dataOutputStream.writeInt(Integer.reverseBytes((int) this.getBlockHeader_Reserved1()));
        });
    }
}

