/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for finalization round
**/
public class FinalizationRoundBuilder implements Serializer {

    /** Finalization epoch. **/
    private final FinalizationEpochDto epoch;

    /** Finalization point. **/
    private final FinalizationPointDto point;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected FinalizationRoundBuilder(DataInputStream stream) {
        try {
            this.epoch = FinalizationEpochDto.loadFromBinary(stream);
            this.point = FinalizationPointDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of FinalizationRoundBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of FinalizationRoundBuilder.
     */
    public static FinalizationRoundBuilder loadFromBinary(DataInputStream stream) {
        return new FinalizationRoundBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param epoch Finalization epoch.
    * @param point Finalization point.
    */
    protected FinalizationRoundBuilder(FinalizationEpochDto epoch, FinalizationPointDto point) {
        GeneratorUtils.notNull(epoch, "epoch is null");
        GeneratorUtils.notNull(point, "point is null");
        this.epoch = epoch;
        this.point = point;
    }
    
    /**
     * Creates an instance of FinalizationRoundBuilder.
     *
     * @param epoch Finalization epoch.
     * @param point Finalization point.
     * @return Instance of FinalizationRoundBuilder.
     */
    public static FinalizationRoundBuilder create(FinalizationEpochDto epoch, FinalizationPointDto point) {
        return new FinalizationRoundBuilder(epoch, point);
    }

    /**
     * Gets finalization epoch.
     *
     * @return Finalization epoch.
     */
    public FinalizationEpochDto getEpoch() {
        return this.epoch;
    }

    /**
     * Gets finalization point.
     *
     * @return Finalization point.
     */
    public FinalizationPointDto getPoint() {
        return this.point;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.epoch.getSize();
        size += this.point.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.epoch);
            GeneratorUtils.writeEntity(dataOutputStream, this.point);
        });
    }
}

