/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Pinned voting key
**/
public class PinnedVotingKeyBuilder implements Serializer {

    /** Voting key. **/
    private final VotingKeyDto votingKey;

    /** Start finalization point. **/
    private final FinalizationPointDto startPoint;

    /** End finalization point. **/
    private final FinalizationPointDto endPoint;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected PinnedVotingKeyBuilder(DataInputStream stream) {
        try {
            this.votingKey = VotingKeyDto.loadFromBinary(stream);
            this.startPoint = FinalizationPointDto.loadFromBinary(stream);
            this.endPoint = FinalizationPointDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of PinnedVotingKeyBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of PinnedVotingKeyBuilder.
     */
    public static PinnedVotingKeyBuilder loadFromBinary(DataInputStream stream) {
        return new PinnedVotingKeyBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param votingKey Voting key.
    * @param startPoint Start finalization point.
    * @param endPoint End finalization point.
    */
    protected PinnedVotingKeyBuilder(VotingKeyDto votingKey, FinalizationPointDto startPoint, FinalizationPointDto endPoint) {
        GeneratorUtils.notNull(votingKey, "votingKey is null");
        GeneratorUtils.notNull(startPoint, "startPoint is null");
        GeneratorUtils.notNull(endPoint, "endPoint is null");
        this.votingKey = votingKey;
        this.startPoint = startPoint;
        this.endPoint = endPoint;
    }
    
    /**
     * Creates an instance of PinnedVotingKeyBuilder.
     *
     * @param votingKey Voting key.
     * @param startPoint Start finalization point.
     * @param endPoint End finalization point.
     * @return Instance of PinnedVotingKeyBuilder.
     */
    public static PinnedVotingKeyBuilder create(VotingKeyDto votingKey, FinalizationPointDto startPoint, FinalizationPointDto endPoint) {
        return new PinnedVotingKeyBuilder(votingKey, startPoint, endPoint);
    }

    /**
     * Gets voting key.
     *
     * @return Voting key.
     */
    public VotingKeyDto getVotingKey() {
        return this.votingKey;
    }

    /**
     * Gets start finalization point.
     *
     * @return Start finalization point.
     */
    public FinalizationPointDto getStartPoint() {
        return this.startPoint;
    }

    /**
     * Gets end finalization point.
     *
     * @return End finalization point.
     */
    public FinalizationPointDto getEndPoint() {
        return this.endPoint;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.votingKey.getSize();
        size += this.startPoint.getSize();
        size += this.endPoint.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.votingKey);
            GeneratorUtils.writeEntity(dataOutputStream, this.startPoint);
            GeneratorUtils.writeEntity(dataOutputStream, this.endPoint);
        });
    }
}

