/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for account restrictions
**/
public class AccountRestrictionsBuilder implements Serializer {

    /** Address on which restrictions are placed. **/
    private final AddressDto address;

    /** Account restrictions. **/
    private final List<AccountRestrictionsInfoBuilder> restrictions;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected AccountRestrictionsBuilder(DataInputStream stream) {
        try {
            this.address = AddressDto.loadFromBinary(stream);
            final long restrictionsCount = Long.reverseBytes(stream.readLong());
            this.restrictions = GeneratorUtils.loadFromBinaryArray(AccountRestrictionsInfoBuilder::loadFromBinary, stream, restrictionsCount);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of AccountRestrictionsBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of AccountRestrictionsBuilder.
     */
    public static AccountRestrictionsBuilder loadFromBinary(DataInputStream stream) {
        return new AccountRestrictionsBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param address Address on which restrictions are placed.
    * @param restrictions Account restrictions.
    */
    protected AccountRestrictionsBuilder(AddressDto address, List<AccountRestrictionsInfoBuilder> restrictions) {
        GeneratorUtils.notNull(address, "address is null");
        GeneratorUtils.notNull(restrictions, "restrictions is null");
        this.address = address;
        this.restrictions = restrictions;
    }
    
    /**
     * Creates an instance of AccountRestrictionsBuilder.
     *
     * @param address Address on which restrictions are placed.
     * @param restrictions Account restrictions.
     * @return Instance of AccountRestrictionsBuilder.
     */
    public static AccountRestrictionsBuilder create(AddressDto address, List<AccountRestrictionsInfoBuilder> restrictions) {
        return new AccountRestrictionsBuilder(address, restrictions);
    }

    /**
     * Gets address on which restrictions are placed.
     *
     * @return Address on which restrictions are placed.
     */
    public AddressDto getAddress() {
        return this.address;
    }

    /**
     * Gets account restrictions.
     *
     * @return Account restrictions.
     */
    public List<AccountRestrictionsInfoBuilder> getRestrictions() {
        return this.restrictions;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.address.getSize();
        size += 8; // restrictionsCount
        size += this.restrictions.stream().mapToInt(o -> o.getSize()).sum();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.address);
            dataOutputStream.writeLong(Long.reverseBytes((long) GeneratorUtils.getSize(this.getRestrictions())));
            GeneratorUtils.writeList(dataOutputStream, this.restrictions);
        });
    }
}

