/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for non-historical root namespace history
**/
public class RootNamespaceHistoryBuilder implements Serializer {

    /** Id of the root namespace history. **/
    private final NamespaceIdDto id;

    /** Namespace owner public key. **/
    private final KeyDto ownerPublicKey;

    /** Lifetime in blocks. **/
    private final NamespaceLifetimeBuilder lifetime;

    /** Root namespace alias. **/
    private final NamespaceAliasTypeDto rootAlias;

    /** Save child sub-namespace paths. **/
    private final List<NamespacePathBuilder> paths;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected RootNamespaceHistoryBuilder(DataInputStream stream) {
        try {
            this.id = NamespaceIdDto.loadFromBinary(stream);
            this.ownerPublicKey = KeyDto.loadFromBinary(stream);
            this.lifetime = NamespaceLifetimeBuilder.loadFromBinary(stream);
            this.rootAlias = NamespaceAliasTypeDto.loadFromBinary(stream);
            final long childrenCount = Long.reverseBytes(stream.readLong());
            this.paths = GeneratorUtils.loadFromBinaryArray(NamespacePathBuilder::loadFromBinary, stream, childrenCount);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of RootNamespaceHistoryBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of RootNamespaceHistoryBuilder.
     */
    public static RootNamespaceHistoryBuilder loadFromBinary(DataInputStream stream) {
        return new RootNamespaceHistoryBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param id Id of the root namespace history.
    * @param ownerPublicKey Namespace owner public key.
    * @param lifetime Lifetime in blocks.
    * @param rootAlias Root namespace alias.
    * @param paths Save child sub-namespace paths.
    */
    protected RootNamespaceHistoryBuilder(NamespaceIdDto id, KeyDto ownerPublicKey, NamespaceLifetimeBuilder lifetime, NamespaceAliasTypeDto rootAlias, List<NamespacePathBuilder> paths) {
        GeneratorUtils.notNull(id, "id is null");
        GeneratorUtils.notNull(ownerPublicKey, "ownerPublicKey is null");
        GeneratorUtils.notNull(lifetime, "lifetime is null");
        GeneratorUtils.notNull(rootAlias, "rootAlias is null");
        GeneratorUtils.notNull(paths, "paths is null");
        this.id = id;
        this.ownerPublicKey = ownerPublicKey;
        this.lifetime = lifetime;
        this.rootAlias = rootAlias;
        this.paths = paths;
    }
    
    /**
     * Creates an instance of RootNamespaceHistoryBuilder.
     *
     * @param id Id of the root namespace history.
     * @param ownerPublicKey Namespace owner public key.
     * @param lifetime Lifetime in blocks.
     * @param rootAlias Root namespace alias.
     * @param paths Save child sub-namespace paths.
     * @return Instance of RootNamespaceHistoryBuilder.
     */
    public static RootNamespaceHistoryBuilder create(NamespaceIdDto id, KeyDto ownerPublicKey, NamespaceLifetimeBuilder lifetime, NamespaceAliasTypeDto rootAlias, List<NamespacePathBuilder> paths) {
        return new RootNamespaceHistoryBuilder(id, ownerPublicKey, lifetime, rootAlias, paths);
    }

    /**
     * Gets id of the root namespace history.
     *
     * @return Id of the root namespace history.
     */
    public NamespaceIdDto getId() {
        return this.id;
    }

    /**
     * Gets namespace owner public key.
     *
     * @return Namespace owner public key.
     */
    public KeyDto getOwnerPublicKey() {
        return this.ownerPublicKey;
    }

    /**
     * Gets lifetime in blocks.
     *
     * @return Lifetime in blocks.
     */
    public NamespaceLifetimeBuilder getLifetime() {
        return this.lifetime;
    }

    /**
     * Gets root namespace alias.
     *
     * @return Root namespace alias.
     */
    public NamespaceAliasTypeDto getRootAlias() {
        return this.rootAlias;
    }

    /**
     * Gets save child sub-namespace paths.
     *
     * @return Save child sub-namespace paths.
     */
    public List<NamespacePathBuilder> getPaths() {
        return this.paths;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.id.getSize();
        size += this.ownerPublicKey.getSize();
        size += this.lifetime.getSize();
        size += this.rootAlias.getSize();
        size += 8; // childrenCount
        size += this.paths.stream().mapToInt(o -> o.getSize()).sum();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.id);
            GeneratorUtils.writeEntity(dataOutputStream, this.ownerPublicKey);
            GeneratorUtils.writeEntity(dataOutputStream, this.lifetime);
            GeneratorUtils.writeEntity(dataOutputStream, this.rootAlias);
            dataOutputStream.writeLong(Long.reverseBytes((long) GeneratorUtils.getSize(this.getPaths())));
            GeneratorUtils.writeList(dataOutputStream, this.paths);
        });
    }
}

