/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for an account key link transaction
**/
public class AccountKeyLinkTransactionBodyBuilder implements Serializer {

    /** Remote public key. **/
    private final KeyDto remotePublicKey;

    /** Link action. **/
    private final LinkActionDto linkAction;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected AccountKeyLinkTransactionBodyBuilder(DataInputStream stream) {
        try {
            this.remotePublicKey = KeyDto.loadFromBinary(stream);
            this.linkAction = LinkActionDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of AccountKeyLinkTransactionBodyBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of AccountKeyLinkTransactionBodyBuilder.
     */
    public static AccountKeyLinkTransactionBodyBuilder loadFromBinary(DataInputStream stream) {
        return new AccountKeyLinkTransactionBodyBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param remotePublicKey Remote public key.
    * @param linkAction Link action.
    */
    protected AccountKeyLinkTransactionBodyBuilder(KeyDto remotePublicKey, LinkActionDto linkAction) {
        GeneratorUtils.notNull(remotePublicKey, "remotePublicKey is null");
        GeneratorUtils.notNull(linkAction, "linkAction is null");
        this.remotePublicKey = remotePublicKey;
        this.linkAction = linkAction;
    }
    
    /**
     * Creates an instance of AccountKeyLinkTransactionBodyBuilder.
     *
     * @param remotePublicKey Remote public key.
     * @param linkAction Link action.
     * @return Instance of AccountKeyLinkTransactionBodyBuilder.
     */
    public static AccountKeyLinkTransactionBodyBuilder create(KeyDto remotePublicKey, LinkActionDto linkAction) {
        return new AccountKeyLinkTransactionBodyBuilder(remotePublicKey, linkAction);
    }

    /**
     * Gets remote public key.
     *
     * @return Remote public key.
     */
    public KeyDto getRemotePublicKey() {
        return this.remotePublicKey;
    }

    /**
     * Gets link action.
     *
     * @return Link action.
     */
    public LinkActionDto getLinkAction() {
        return this.linkAction;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.remotePublicKey.getSize();
        size += this.linkAction.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.remotePublicKey);
            GeneratorUtils.writeEntity(dataOutputStream, this.linkAction);
        });
    }
}

