package io.naradrama.prologue.grpc.cqrs;

import io.naradrama.prologue.domain.Offset;
import io.naradrama.prologue.domain.cqrs.DomainMessageType;
import io.naradrama.prologue.domain.cqrs.FailureMessage;
import io.naradrama.prologue.domain.cqrs.TraceHeader;
import io.naradrama.prologue.domain.cqrs.command.CommandResponse;
import io.naradrama.prologue.domain.cqrs.command.CqrsBaseCommand;
import io.naradrama.prologue.domain.cqrs.command.CqrsBaseCommandType;
import io.naradrama.prologue.domain.cqrs.command.CqrsCommandType;
import io.naradrama.prologue.domain.cqrs.command.CqrsUserCommand;
import io.naradrama.prologue.domain.cqrs.query.*;
import io.naradrama.prologue.domain.cqrs.query.dynamic.Connector;
import io.naradrama.prologue.domain.cqrs.query.dynamic.Operator;
import io.naradrama.prologue.domain.cqrs.query.dynamic.QueryParam;
import io.naradrama.prologue.domain.cqrs.query.dynamic.QueryParams;
import io.naradrama.prologue.util.string.StringUtil;
import io.naradrama.protobuf.OffsetMsg;
import io.naradrama.protobuf.SortDirectionMsg;
import io.naradrama.protobuf.cqrs.DomainMessageTypeMsg;
import io.naradrama.protobuf.cqrs.FailureMessageMsg;
import io.naradrama.protobuf.cqrs.TraceHeaderMsg;
import io.naradrama.protobuf.cqrs.command.*;
import io.naradrama.protobuf.cqrs.query.CqrsBaseQueryMsg;
import io.naradrama.protobuf.cqrs.query.CqrsDynamicQueryMsg;
import io.naradrama.protobuf.cqrs.query.CqrsQueryTypeMsg;
import io.naradrama.protobuf.cqrs.query.CqrsUserQueryMsg;
import io.naradrama.protobuf.cqrs.query.dynamic.ConnectorMsg;
import io.naradrama.protobuf.cqrs.query.dynamic.OperatorMsg;
import io.naradrama.protobuf.cqrs.query.dynamic.QueryParamMsg;
import io.naradrama.protobuf.cqrs.query.dynamic.QueryParamsMsg;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

public class GrpcCqrsMessageUtils {
    //
    public static CqrsUserCommandMsg buildCqrsCommandMsg(CqrsUserCommand cqrsUserCommand) {
        //
        return CqrsUserCommandMsg.newBuilder()
                .setId(cqrsUserCommand.getId())
                .setDomainMessageType(buildDomainMessageTypeMsg(cqrsUserCommand.getDomainMessageType()))
                .setTraceHeader(buildTraceHeader(cqrsUserCommand.getTraceHeader()))
                .setCqrsCommandType(buildCqrsCommandTypeMsg(cqrsUserCommand.getCqrsCommandType()))
                .setRequestTime(cqrsUserCommand.getRequestTime())
                .setWaitingTime(cqrsUserCommand.getWaitingTime())
                .setRequestFailed(cqrsUserCommand.isRequestFailed())
                .setFailureMessage(buildFailureMessageMsg(cqrsUserCommand.getFailureMessage()))
                .setCommandResponse(buildCommandResponseMsg(cqrsUserCommand.getCommandResponse()))
                .build();
    }

    public static CqrsBaseCommandMsg buildCqrsCommandMsg(CqrsBaseCommand cqrsBaseCommand) {
        //
        return CqrsBaseCommandMsg.newBuilder()
                .setId(cqrsBaseCommand.getId())
                .setDomainMessageType(buildDomainMessageTypeMsg(cqrsBaseCommand.getDomainMessageType()))
                .setTraceHeader(buildTraceHeader(cqrsBaseCommand.getTraceHeader()))
                .setCqrsCommandType(buildCqrsCommandTypeMsg(cqrsBaseCommand.getCqrsCommandType()))
                .setRequestTime(cqrsBaseCommand.getRequestTime())
                .setWaitingTime(cqrsBaseCommand.getWaitingTime())
                .setRequestFailed(cqrsBaseCommand.isRequestFailed())
                .setFailureMessage(buildFailureMessageMsg(cqrsBaseCommand.getFailureMessage()))
                .setCommandResponse(buildCommandResponseMsg(cqrsBaseCommand.getCommandResponse()))
                .setCqrsBaseCommandType(buildCqrsBaseCommandTypeMsg(cqrsBaseCommand.getCqrsBaseCommandType()))
                .build();
    }

    public static CqrsBaseQueryMsg buildCqrsQueryMsg(CqrsBaseQuery cqrsBaseQuery) {
        //
        return CqrsBaseQueryMsg.newBuilder()
                .setId(cqrsBaseQuery.getId())
                .setDomainMessageType(buildDomainMessageTypeMsg(cqrsBaseQuery.getDomainMessageType()))
                .setTraceHeader(buildTraceHeader(cqrsBaseQuery.getTraceHeader()))
                .setCqrsQueryType(buildCqrsQueryTypeMsg(cqrsBaseQuery.getCqrsQueryType()))
                .setRequestTime(cqrsBaseQuery.getRequestTime())
                .setWaitingTime(cqrsBaseQuery.getWaitingTime())
                .setRequestFailed(cqrsBaseQuery.isRequestFailed())
                .setFailureMessage(buildFailureMessageMsg(cqrsBaseQuery.getFailureMessage()))
                .build();
    }

    public static CqrsUserQueryMsg buildCqrsQueryMsg(CqrsUserQuery cqrsUserQuery) {
        //
        return CqrsUserQueryMsg.newBuilder()
                .setId(cqrsUserQuery.getId())
                .setDomainMessageType(buildDomainMessageTypeMsg(cqrsUserQuery.getDomainMessageType()))
                .setTraceHeader(buildTraceHeader(cqrsUserQuery.getTraceHeader()))
                .setCqrsQueryType(buildCqrsQueryTypeMsg(cqrsUserQuery.getCqrsQueryType()))
                .setRequestTime(cqrsUserQuery.getRequestTime())
                .setWaitingTime(cqrsUserQuery.getWaitingTime())
                .setRequestFailed(cqrsUserQuery.isRequestFailed())
                .setFailureMessage(buildFailureMessageMsg(cqrsUserQuery.getFailureMessage()))
                .build();
    }

    public static CqrsDynamicQueryMsg buildCqrsQueryMsg(CqrsDynamicQuery cqrsDynamicQuery) {
        //
        return CqrsDynamicQueryMsg.newBuilder()
                .setId(cqrsDynamicQuery.getId())
                .setDomainMessageType(buildDomainMessageTypeMsg(cqrsDynamicQuery.getDomainMessageType()))
                .setTraceHeader(buildTraceHeader(cqrsDynamicQuery.getTraceHeader()))
                .setCqrsQueryType(buildCqrsQueryTypeMsg(cqrsDynamicQuery.getCqrsQueryType()))
                .setRequestTime(cqrsDynamicQuery.getRequestTime())
                .setWaitingTime(cqrsDynamicQuery.getWaitingTime())
                .setRequestFailed(cqrsDynamicQuery.isRequestFailed())
                .setFailureMessage(buildFailureMessageMsg(cqrsDynamicQuery.getFailureMessage()))
                .setQueryParams(buildQueryParamsMsg(cqrsDynamicQuery.getQueryParams()))
                .setOffset(buildOffsetMsg(cqrsDynamicQuery.getOffset()))
                .build();
    }

    // ----------------------------------------------------------------------------------------------

    private static CommandResponseMsg buildCommandResponseMsg(CommandResponse commandResponse) {
        //
        return CommandResponseMsg.newBuilder()
                .addAllEntityIds(commandResponse.getEntityIds())
                .setResult(commandResponse.isResult())
                .build();
    }
    private static TraceHeaderMsg buildTraceHeader(TraceHeader traceHeader) {
        //
        return TraceHeaderMsg.newBuilder()
                .setTraceId(traceHeader.getTraceId())
                .setUserId(traceHeader.getUserId())
                .setParentService(traceHeader.getParentService())
                .setService(traceHeader.getService())
                .setParentMessage(traceHeader.getParentMessage())
                .setMessage(traceHeader.getMessage())
                .setTime(traceHeader.getTime())
                .build();
    }

    private static QueryParamsMsg buildQueryParamsMsg(QueryParams queryParams) {
        //
        return QueryParamsMsg.newBuilder()
                .addAllSpecialOperator(buildOperatorMsgList(queryParams.getSpecialOperator()))
                .addAllQueryParams(buildQueryParamMsgList(queryParams.getQueryParams()))
                .build();
    }

    private static List<OperatorMsg> buildOperatorMsgList(List<Operator> specialOperator) {
        //
        if (specialOperator == null) {
            return Collections.emptyList();
        }
        return specialOperator.stream()
                .map(GrpcCqrsMessageUtils::buildOperatorMsg)
                .collect(Collectors.toList());
    }

    private static List<QueryParamMsg> buildQueryParamMsgList(List<QueryParam> queryParams) {
        //
        if (queryParams == null) {
            return Collections.emptyList();
        }
        return queryParams.stream()
                .map(GrpcCqrsMessageUtils::buildQueryParamMsg)
                .collect(Collectors.toList());
    }

    private static QueryParamMsg buildQueryParamMsg(QueryParam queryParam) {
        //
        return QueryParamMsg.newBuilder()
                .setFieldName(queryParam.getFieldName())
                .setOperator(buildOperatorMsg(queryParam.getOperation()))
                .setValue(queryParam.getValue())
                .setConnector(buildConnectorMsg(queryParam.getConnector()))
                .build();
    }

    private static OffsetMsg buildOffsetMsg(Offset offset) {
        //
        return OffsetMsg.newBuilder()
                .setOffset(offset.getOffset())
                .setLimit(offset.getLimit())
                .setTotalCount(offset.getTotalCount())
                .setTotalCountRequested(offset.isTotalCountRequested())
                .setSortingField(offset.getSortingField())
                .setSortDirection(buildSortDirectionMsg(offset.getSortDirection()))
                .build();
    }


    private static CqrsCommandTypeMsg buildCqrsCommandTypeMsg(CqrsCommandType cqrsCommandType) {
        //
        return CqrsCommandTypeMsg.valueOf(cqrsCommandType.name());
    }

    private static DomainMessageTypeMsg buildDomainMessageTypeMsg(DomainMessageType domainMessageType) {
        //
        return DomainMessageTypeMsg.valueOf(domainMessageType.name());
    }

    private static CqrsBaseCommandTypeMsg buildCqrsBaseCommandTypeMsg(CqrsBaseCommandType cqrsBaseCommandType) {
        //
        return CqrsBaseCommandTypeMsg.valueOf(cqrsBaseCommandType.name());
    }

    private static CqrsQueryTypeMsg buildCqrsQueryTypeMsg(CqrsQueryType cqrsQueryType) {
        //
        return CqrsQueryTypeMsg.valueOf(cqrsQueryType.name());
    }

    private static ConnectorMsg buildConnectorMsg(Connector connector) {
        //
        return ConnectorMsg.valueOf(connector.name());
    }

    private static OperatorMsg buildOperatorMsg(Operator operator) {
        //
        return OperatorMsg.valueOf(operator.name());
    }

    private static SortDirectionMsg buildSortDirectionMsg(Offset.SortDirection sortDirection) {
        //
        return SortDirectionMsg.valueOf(sortDirection.name());
    }

    public static FailureMessage fromMessage(FailureMessageMsg failureMessageMsg) {
        //
        return new FailureMessage(
                failureMessageMsg.getExceptionName(),
                failureMessageMsg.getExceptionMessage(),
                failureMessageMsg.getExceptionCode());
    }

    public static FailureMessageMsg buildFailureMessageMsg(FailureMessage failureMessage) {
        //
        FailureMessageMsg.Builder builder = FailureMessageMsg.newBuilder();
        if (failureMessage == null) {
            return builder.build();
        }
        if (!StringUtil.isEmpty(failureMessage.getExceptionName())) {
            builder.setExceptionName(failureMessage.getExceptionName());
        }
        if (!StringUtil.isEmpty(failureMessage.getExceptionMessage())) {
            builder.setExceptionMessage(failureMessage.getExceptionMessage());
        }
        if (!StringUtil.isEmpty(failureMessage.getExceptionCode())) {
            builder.setExceptionCode(failureMessage.getExceptionCode());
        }
        return builder.build();
    }

    public static CommandResponse fromMessage(CommandResponseMsg commandResponseMsg) {
        //
        if(commandResponseMsg == null ){
            return null;
        }
        CommandResponse commandResponse = new CommandResponse();
        commandResponse.setEntityIds(commandResponseMsg.getEntityIdsList());
        commandResponse.setResult(commandResponseMsg.getResult());

        return commandResponse;
    }

}
