package io.naradrama.prologue.grpc;

import io.naradrama.prologue.domain.IdName;
import io.naradrama.prologue.domain.NameValue;
import io.naradrama.prologue.domain.NameValueList;
import io.naradrama.prologue.domain.granule.Name;
import io.naradrama.prologue.domain.lang.LangName;
import io.naradrama.prologue.domain.lang.LangNames;
import io.naradrama.prologue.domain.lang.LangString;
import io.naradrama.prologue.domain.lang.LangStrings;
import io.naradrama.prologue.domain.workspace.Workspace;
import io.naradrama.prologue.domain.workspace.WorkspaceType;
import io.naradrama.protobuf.BooleanArrayMsg;
import io.naradrama.protobuf.IdNameMsg;
import io.naradrama.protobuf.NameValueListMsg;
import io.naradrama.protobuf.NameValueMsg;
import io.naradrama.protobuf.granule.NameMsg;
import io.naradrama.protobuf.lang.LangNameMsg;
import io.naradrama.protobuf.lang.LangNamesMsg;
import io.naradrama.protobuf.lang.LangStringMsg;
import io.naradrama.protobuf.lang.LangStringsMsg;
import io.naradrama.protobuf.nara.WorkspaceMsg;
import io.naradrama.protobuf.nara.WorkspaceTypeMsg;

import java.util.*;
import java.util.stream.Collectors;

public class PrologueGrpcMessage {
    //
    public static LangStringMsg toMessage(LangString langString) {
        //
        LangStringMsg.Builder builder = LangStringMsg.newBuilder();
        builder.setLang(langString.getLanguage());
        builder.setString(langString.getString());

        return builder.build();
    }

    public static List<LangStringMsg> toMessage(List<LangString> langStrings) {
        //
        return langStrings.stream()
                .map(PrologueGrpcMessage::toMessage)
                .collect(Collectors.toList());
    }

    public static LangStringsMsg toMessage(LangStrings langStrings) {
        //
        LangStringsMsg.Builder builder = LangStringsMsg.newBuilder();
        builder.setDefaultLanguage(langStrings.getDefaultLanguage());
        builder.addAllLangStrings(langStrings.getLangStrings()
                .stream()
                    .map(PrologueGrpcMessage::toMessage)
                    .collect(Collectors.toList()));


        return builder.build();
    }

    public static LangString fromMessage(LangStringMsg langStringMsg) {
        //
        return LangString.newString(langStringMsg.getLang(), langStringMsg.getString());
    }

    public static List<LangString> fromMessage(List<LangStringMsg> langStringMsgs) {
        //
        return langStringMsgs.stream()
                .map(PrologueGrpcMessage::fromMessage)
                .collect(Collectors.toList());
    }

    public static LangStrings fromMessage(LangStringsMsg langStringsMsg) {
        //
        LangStrings langStrings = new LangStrings();
        langStrings.setDefaultLanguage(langStringsMsg.getDefaultLanguage());
        langStrings.setLangStrings(langStringsMsg.getLangStringsList()
                .stream()
                    .map(PrologueGrpcMessage::fromMessage)
                    .collect(Collectors.toList()));

        return langStrings;
    }

    public static IdNameMsg toIdNameMsg(IdName idName) {
        //
        if (idName == null) {
            return IdNameMsg.newBuilder()
                    .setId("")
                    .setName("")
                    .build();
        }
        return IdNameMsg.newBuilder()
                .setId(idName.getId())
                .setName(idName.getName())
                .build();
    }

    public static IdName toIdName(IdNameMsg message) {
        //
        return new IdName(message.getId(), message.getName());
    }

    public static NameValueMsg toNameValueMsg(NameValue nameValue) {
        //
        return NameValueMsg.newBuilder().setName(nameValue.getName()).setValue(nameValue.getValue()).build();
    }

    public static NameValue toNameValue(NameValueMsg nameValueMsg) {
        //
        return new NameValue(nameValueMsg.getName(), nameValueMsg.getValue());
    }

    public static NameValueList toNameValueList(List<NameValueMsg> nameValuesList) {
        //
        NameValueList nameValues = new NameValueList();
        nameValuesList.forEach(nameValue -> nameValues.add(nameValue.getName(), nameValue.getValue()));
        return nameValues;
    }

    public static NameValueList toNameValueList(NameValueListMsg message) {
        //
        NameValueList nameValueList = new NameValueList();
        List<NameValue> nameValues = message.getNameValuesList().stream().map(PrologueGrpcMessage::toNameValue).collect(Collectors.toList());
        nameValueList.setNameValues(nameValues);
        return nameValueList;
    }

    public static Workspace toWorkspace(WorkspaceMsg workspaceMsg) {
        //
        Workspace workspace = new Workspace(
                workspaceMsg.getId(),
                workspaceMsg.getName(),
                workspaceMsg.getTenantId()
        );
        if( workspaceMsg.getRolesList() != null ) {
            List<String> roles = new ArrayList<>();
            workspaceMsg.getRolesList().forEach(role -> roles.add(role));
            workspace.setRoles(roles);
        }

        return workspace;
    }

    public static WorkspaceType toWorkspaceType(WorkspaceTypeMsg workspaceTypeMsg) {
        //
        return WorkspaceType.valueOf(workspaceTypeMsg.name());
    }

    public static Name toName(NameMsg nameMsg) {
        //
        return new Name(nameMsg.getGivenName(), nameMsg.getMiddleName(), nameMsg.getFamilyName());
    }

    public static LangName toLangName(LangNameMsg langNameMsg) {
        //
        return new LangName(langNameMsg.getLanguage(), toName(langNameMsg.getName()));
    }

    public static List<LangName> toLangNameList(List<LangNameMsg> langNameMsgs) {
        //
        return langNameMsgs.stream()
                .map(PrologueGrpcMessage::toLangName)
                .collect(Collectors.toList());
    }

    public static LangNames toLangNames(LangNamesMsg langNamesMsg) {
        //
        LangNames langNames = new LangNames();
        langNames.setDefaultLanguage(langNamesMsg.getDefaultLanguage());

        Map<String, Name> nameMap = new HashMap<>();
        langNamesMsg.getNameMapMap().forEach((key, value)-> nameMap.put(key, toName(value)));

        langNames.setNameMap(nameMap);

        return langNames;
    }

    public static WorkspaceMsg toWorkspaceMsg(Workspace workspace) {
        //
        WorkspaceMsg.Builder builder
                = WorkspaceMsg.newBuilder()
                .setId(workspace.getId())
                .setName(workspace.getName())
                .setTenantId(workspace.getTenantId());

        if( workspace.getRoles() != null ) {
            workspace.getRoles().forEach(role -> builder.addRoles(role));
        }

        return builder.build();
    }

    public static WorkspaceTypeMsg toWorkspaceTypeMsg(WorkspaceType workspaceType) {
        //
        return WorkspaceTypeMsg.valueOf(workspaceType.name());
    }

    public static NameMsg toNameMsg(Name name) {
        //
        NameMsg.Builder builder
                = NameMsg.newBuilder()
                .setGivenName(Optional.ofNullable(name.getGivenName()).orElse(""))
                .setMiddleName(Optional.ofNullable(name.getMiddleName()).orElse(""))
                .setFamilyName(Optional.ofNullable(name.getFamilyName()).orElse(""));

        return builder.build();
    }

    public static LangNameMsg toLangNameMsg(LangName langName) {
        //
        LangNameMsg.Builder builder
                = LangNameMsg.newBuilder()
                .setLanguage(langName.getLanguage())
                .setName(toNameMsg(langName.getName()));

        return builder.build();
    }

    public static LangNamesMsg toLangNamesMsg(LangNames langNames) {
        //
        LangNamesMsg.Builder builder = LangNamesMsg.newBuilder();
        builder.setDefaultLanguage(langNames.getDefaultLanguage());

        Map<String, NameMsg> nameMsgMap = new HashMap<>();
        langNames.getNameMap().forEach((key, value)-> nameMsgMap.put(key, toNameMsg(value)));

        builder.putAllNameMap(nameMsgMap);

        return builder.build();
    }

    public static BooleanArrayMsg toMessage(boolean[] values) {
        //
        BooleanArrayMsg.Builder builder = BooleanArrayMsg.newBuilder();
        for (int i = 0; i < values.length; i++) {
            builder.addValues(values[i]);
        }
        return builder.build();
    }

    public static boolean [] fromMessage(BooleanArrayMsg booleanArrayMsg) {
        //
        int valuesCount = booleanArrayMsg.getValuesCount();
        boolean[] values = new boolean[valuesCount];
        for (int i = 0; i < valuesCount; i++) {
            values[i] = booleanArrayMsg.getValues(i);
        }
        return values;
    }
}
