package io.meiro.sdk.offline

import io.meiro.sdk.event.Event
import io.meiro.sdk.event.EventAppInfo
import io.meiro.sdk.event.EventDeviceInfo
import io.meiro.sdk.event.EventFirebaseInfo
import io.meiro.sdk.event.EventOsInfo
import io.meiro.sdk.event.EventType
import io.meiro.sdk.event.EventUserInfo
import java.time.Instant

internal class DbEventMapping {

    fun toDbEvent(event: Event): DbEvent {
        return DbEvent(
            id = 0,
            type = event.type.id,
            timestamp = event.timestamp.toEpochMilli(),
            version = event.version,
            propertiesSerialized = serializeProperties(event.properties),
            appInfoId = event.appInfo.id,
            appInfoName = event.appInfo.name,
            appInfoVersion = event.appInfo.version,
            appInfoLanguage = event.appInfo.language,
            appInfoAdId = event.appInfo.adId,
            osInfoType = event.osInfo.type,
            osInfoVersion = event.osInfo.version,
            deviceInfoManufacturer = event.deviceInfo?.manufacturer ?: "",
            deviceInfoModel = event.deviceInfo?.model ?: "",
            firebaseInfoProjectId = event.firebaseInfo.projectId,
            firebaseInfoToken = event.firebaseInfo.token,
            userInfoUserId = event.userInfo.userId,
            userInfoSessionId = event.userInfo.sessionId
        )
    }

    private fun serializeProperties(properties: Map<String, Any>): String {
        return properties.entries.joinToString(separator = DELIMITER) { "${it.key}=${it.value}" }
    }

    private fun deserializeProperties(serialized: String): Map<String, String> {
        return if (serialized.isEmpty()) {
            emptyMap()
        } else {
            serialized.split(DELIMITER).associate {
                val splits = it.split("=")
                if (splits.size == 2) {
                    splits[0] to splits[1]
                } else {
                    splits[0] to ""
                }
            }
        }
    }

    fun toEvent(dbEvent: DbEvent): Event {
        return Event(
            type = EventType.entries.first { it.id == dbEvent.type },
            version = dbEvent.version,
            properties = deserializeProperties(dbEvent.propertiesSerialized),
            timestamp = Instant.ofEpochMilli(dbEvent.timestamp),
            appInfo = EventAppInfo(
                id = dbEvent.appInfoId,
                name = dbEvent.appInfoName,
                version = dbEvent.appInfoVersion,
                language = dbEvent.appInfoLanguage,
                adId = dbEvent.appInfoAdId
            ),
            osInfo = EventOsInfo(
                type = dbEvent.osInfoType,
                version = dbEvent.osInfoVersion
            ),
            deviceInfo = if (dbEvent.deviceInfoManufacturer.isNotEmpty() && dbEvent.deviceInfoModel.isNotEmpty()) {
                EventDeviceInfo(
                    manufacturer = dbEvent.deviceInfoManufacturer,
                    model = dbEvent.deviceInfoModel
                )
            } else {
                null
            },
            firebaseInfo = EventFirebaseInfo(
                projectId = dbEvent.firebaseInfoProjectId,
                token = dbEvent.firebaseInfoToken
            ),
            userInfo = EventUserInfo(
                userId = dbEvent.userInfoUserId,
                sessionId = dbEvent.userInfoSessionId
            )
        )
    }

    companion object {

        private const val DELIMITER = ";-;"
    }
}
