package io.meiro.sdk.fcm

import android.os.Parcelable
import com.google.firebase.messaging.RemoteMessage
import kotlinx.parcelize.Parcelize
import kotlinx.serialization.json.Json

@Parcelize
internal data class NotificationData(
    val id: String,
    val googleMessageId: String,
    val action: NotificationAction,
    val ui: NotificationUi,
    val payload: Map<String, String> = emptyMap()
) : Parcelable {

    constructor(message: RemoteMessage) : this(
        id = message.data[MESSAGE_ID_KEY] ?: "",
        googleMessageId = message.messageId ?: "",
        ui = NotificationUi(
            title = message.data[TITLE_KEY] ?: "",
            body = message.data[BODY_KEY] ?: "",
            imageUrl = message.data[IMAGE_URL_KEY],
        ),
        action = when (message.data[ACTION_KEY]) {
            APP_ACTION -> NotificationAction.App
            DEEPLINK_ACTION -> if (message.data[URL_KEY]?.isNotEmpty() == true) {
                NotificationAction.DeepLink(message.data[URL_KEY] ?: "")
            } else {
                NotificationAction.App
            }

            BROWSER_ACTION -> if (message.data[URL_KEY]?.isNotEmpty() == true) {
                NotificationAction.Browser(message.data[URL_KEY] ?: "")
            } else {
                NotificationAction.App
            }

            else -> NotificationAction.App
        },
        payload = message.data.filterKeys {
            it != MESSAGE_ID_KEY &&
                it != TITLE_KEY &&
                it != BODY_KEY &&
                it != IMAGE_URL_KEY &&
                it != ACTION_KEY &&
                it != URL_KEY &&
                it != IS_MEIRO_MESSAGE_KEY
        },
    )

    fun toEventProperties(): Map<String, Any> {
        return mapOf(
            MESSAGE_ID_KEY to id,
            TITLE_KEY to ui.title,
            BODY_KEY to ui.body,
            ACTION_KEY to when (action) {
                is NotificationAction.App -> APP_ACTION
                is NotificationAction.DeepLink -> DEEPLINK_ACTION
                is NotificationAction.Browser -> BROWSER_ACTION
            },
            URL_KEY to when (action) {
                is NotificationAction.DeepLink -> action.url
                is NotificationAction.Browser -> action.url
                else -> ""
            },
            IMAGE_URL_KEY to (ui.imageUrl ?: "")
        ) + payload.mapValues { (_, value) ->
            try {
                Json.parseToJsonElement(value)
            } catch (e: Exception) {
                // ignore
                value
            }
        }
    }

    companion object {

        private const val MESSAGE_ID_KEY = "message_id"
        private const val TITLE_KEY = "title"
        private const val BODY_KEY = "body"
        private const val IMAGE_URL_KEY = "image"
        private const val ACTION_KEY = "action"
        private const val URL_KEY = "url"
        private const val APP_ACTION = "app"
        private const val BROWSER_ACTION = "browser"
        private const val DEEPLINK_ACTION = "deeplink"
        internal const val IS_MEIRO_MESSAGE_KEY = "is_meiro_message"
    }
}

@Parcelize
internal data class NotificationUi(
    val title: String,
    val body: String,
    val imageUrl: String?,
) : Parcelable

internal sealed interface NotificationAction : Parcelable {

    @Parcelize
    data object App : NotificationAction

    @Parcelize
    data class DeepLink(val url: String) : NotificationAction

    @Parcelize
    data class Browser(val url: String) : NotificationAction
}
