package io.meiro.sdk.fcm

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import io.meiro.sdk.MeiroSdk
import io.meiro.sdk.event.EventType

/**
 * This Activity is opened every time a user clicks on a notification. It is used to track the
 * event to the SDK and to open either deeplink or a main app. It is closed immediately.
 */
@Suppress("DEPRECATION")
internal open class MeiroPushNotificationsTrackingJumpthroughActivity : Activity() {

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        intent.getParcelableExtra<NotificationData>(EXTRA_NOTIFICATION_DATA)?.let { notificationData ->
            processPushClick(this, notificationData)
            performClickAction(notificationData)
        }
        finish()
    }

    private fun performClickAction(notificationData: NotificationData) {
        startActivity(
            when (val action = notificationData.action) {
                NotificationAction.App -> getLaunchIntent(notificationData.payload)
                is NotificationAction.Browser -> getUrlIntent(action.url, notificationData.payload)
                is NotificationAction.DeepLink -> getUrlIntent(action.url, notificationData.payload)
            }
        )
    }

    open fun processPushClick(
        context: Context,
        data: NotificationData,
    ) {
        MeiroSdk.impl?.trackEventInternal(EventType.FCM_MESSAGE_CLICK, data.toEventProperties())
        MeiroSdk.impl?.log("Push notification clicked")
    }

    private fun getLaunchIntent(payload: Map<String, String>): Intent? {
        return packageManager.getLaunchIntentForPackage(packageName)?.putDataPayloadToExtras(payload)
            ?.addFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT)
    }

    private fun getUrlIntent(url: String, payload: Map<String, String>): Intent {
        return Intent(Intent.ACTION_VIEW, Uri.parse(url)).putDataPayloadToExtras(payload)
            .addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
    }

    private fun Intent.putDataPayloadToExtras(payload: Map<String, String>): Intent {
        payload.forEach { (key, value) ->
            putExtra(key, value)
        }
        return this
    }

    companion object {

        private const val EXTRA_NOTIFICATION_DATA = "notification_data"

        fun getClickIntent(
            context: Context,
            data: NotificationData,
        ): Intent {
            return Intent(context, MeiroPushNotificationsTrackingJumpthroughActivity::class.java)
                .putExtra(EXTRA_NOTIFICATION_DATA, data)
                .addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP or Intent.FLAG_ACTIVITY_CLEAR_TOP)
        }
    }
}
