package io.meiro.sdk.fcm

import android.annotation.SuppressLint
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import androidx.core.app.NotificationCompat
import androidx.core.app.NotificationManagerCompat
import com.google.firebase.messaging.RemoteMessage
import io.meiro.sdk.MeiroSdk
import io.meiro.sdk.event.EventType
import java.io.IOException
import java.net.URL
import kotlin.concurrent.thread
import kotlin.random.Random

object MeiroNotifications {

    private val requestCodeGenerator = Random

    /**
     * Checks whether the given [message] is sent by a Meiro platform.
     */
    fun isMeiroMessage(message: RemoteMessage): Boolean {
        return message.data.containsKey(NotificationData.IS_MEIRO_MESSAGE_KEY)
    }

    /**
     * Show a notification for the given [message].
     *
     * Since API [android.os.Build.VERSION_CODES.TIRAMISU] this methods requires the [android.Manifest.permission.POST_NOTIFICATIONS] permission to be granted
     * to display a notification.
     */
    fun show(
        context: Context,
        message: RemoteMessage
    ) {
        show(
            context = context,
            message = message,
            configuration = MeiroSdk.impl?.requireConfiguration()?.pushNotifications ?: return
        )
    }

    internal fun show(
        context: Context,
        configuration: PushNotificationsConfiguration,
        message: RemoteMessage
    ) {
        if (!configuration.pushEnabled) return

        val data = NotificationData(message)
        MeiroSdk.impl?.log("Received notification: $data")
        MeiroSdk.impl?.trackEventInternal(
            EventType.FCM_MESSAGE_RECEIVED,
            data.toEventProperties()
        )
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.TIRAMISU &&
            context.checkSelfPermission(android.Manifest.permission.POST_NOTIFICATIONS) != PackageManager.PERMISSION_GRANTED
        ) {
            return
        }

        showNotification(
            context = context,
            configuration = configuration,
            data = data
        )
    }

    @SuppressLint("MissingPermission")
    private fun showNotification(
        context: Context,
        configuration: PushNotificationsConfiguration,
        data: NotificationData
    ) {
        val notificationManager = NotificationManagerCompat.from(context)
        ensureNotificationChannelExists(notificationManager, configuration)
        val notificationBuilder = NotificationCompat.Builder(context, configuration.channelId)
            .setContentTitle(data.ui.title)
            .setContentText(data.ui.body)
            .setStyle(NotificationCompat.BigTextStyle().bigText(data.ui.body))
            .setSmallIcon(configuration.pushIconResId ?: android.R.drawable.ic_dialog_info)
            .setPriority(NotificationCompat.PRIORITY_DEFAULT)
            .setAutoCancel(true)
            .apply {
                configuration.accentColor?.let { accentColor ->
                    setColor(accentColor)
                }
            }
        handleNotificationImage(notificationBuilder, data.ui.imageUrl)
        handleNotificationClick(context, notificationBuilder, data)
        notificationManager.notify(data.id.hashCode(), notificationBuilder.build())
    }

    private fun handleNotificationImage(notificationBuilder: NotificationCompat.Builder, imageUrl: String?) {
        if (imageUrl != null) {
            val bitmap = getBitmapFromUrl(imageUrl)
            bitmap?.let {
                notificationBuilder.setStyle(
                    NotificationCompat.BigPictureStyle()
                        .bigPicture(it)
                )
            }
        }
    }

    private fun getBitmapFromUrl(url: String): Bitmap? {
        var bmp: Bitmap? = null
        thread {
            try {
                val input = URL(url).openStream()
                bmp = BitmapFactory.decodeStream(input)
            } catch (e: IOException) {
                MeiroSdk.impl?.log("Failed to download image: $url", e)
            }
        }.join()
        return bmp
    }

    private fun handleNotificationClick(context: Context, notificationBuilder: NotificationCompat.Builder, data: NotificationData) {
        notificationBuilder.setContentIntent(
            getNotificationPendingIntent(context, trackingIntent = getTrackingIntent(context, data), data)
        )
    }

    private fun getTrackingIntent(context: Context, data: NotificationData): Intent {
        return MeiroPushNotificationsTrackingJumpthroughActivity.getClickIntent(context, data)
    }

    private fun getNotificationPendingIntent(context: Context, trackingIntent: Intent, data: NotificationData): PendingIntent {
        return when (data.action) {
            is NotificationAction.App -> {
                PendingIntent.getActivity(
                    context,
                    requestCodeGenerator.nextInt(),
                    trackingIntent,
                    getPendingIntentFlags()
                )
            }

            is NotificationAction.DeepLink -> {
                PendingIntent.getActivity(
                    context,
                    requestCodeGenerator.nextInt(),
                    trackingIntent,
                    getPendingIntentFlags()
                )
            }

            is NotificationAction.Browser -> {
                PendingIntent.getActivity(
                    context,
                    requestCodeGenerator.nextInt(),
                    trackingIntent,
                    getPendingIntentFlags()
                )
            }
        }
    }

    private fun getPendingIntentFlags(): Int {
        return PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
    }

    private fun ensureNotificationChannelExists(notificationManagerCompat: NotificationManagerCompat, configuration: PushNotificationsConfiguration) {
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O &&
            notificationManagerCompat.getNotificationChannel(configuration.channelId) == null
        ) {
            val channel = NotificationChannel(
                configuration.channelId,
                configuration.channelName,
                NotificationManager.IMPORTANCE_DEFAULT
            ).apply {
                description = configuration.channelDescription
            }
            notificationManagerCompat.createNotificationChannel(channel)
        }
    }
}
