package io.meiro.sdk.offline

import android.content.Context
import androidx.room.Room
import androidx.work.ExistingWorkPolicy
import androidx.work.WorkManager
import io.meiro.sdk.MeiroLogger
import io.meiro.sdk.api.MeiroEventsApi
import io.meiro.sdk.event.Event
import io.meiro.sdk.offline.workmanager.SyncWorker
import java.time.Duration
import java.time.Instant

/**
 * Manages syncing events with the server.
 */
internal class SyncManager(
    private val context: Context,
    private val api: MeiroEventsApi,
    private val logger: MeiroLogger
) {

    private val database = Room.databaseBuilder(
        context,
        MeiroEventsDatabase::class.java,
        DB_NAME
    ).build()
    private val dbMapping = DbEventMapping()

    private val dao get() = database.eventDao()

    suspend fun sync(now: Instant = Instant.now()) {
        removeOldEvents(now)
        syncAndDeleteEvents()
    }

    private suspend fun syncAndDeleteEvents() {
        val events = dao.getAll()
        events.forEach {
            api.sendEvent(dbMapping.toEvent(it))
            dao.deleteById(it.id)
        }
    }

    private suspend fun removeOldEvents(now: Instant) {
        val timestamp = now.minus(EVENT_LIFETIME).toEpochMilli()
        dao.deleteOlderThan(timestamp)
    }

    suspend fun saveEvent(event: Event) {
        dao.insert(dbMapping.toDbEvent(event))
        planSyncOnConnection()
    }

    private fun planSyncOnConnection() {
        logger.log("Planning sync on connection")
        WorkManager.getInstance(context).enqueueUniqueWork(
            SyncWorker.WORK_NAME,
            ExistingWorkPolicy.REPLACE,
            SyncWorker.createWorkRequest()
        )
    }

    companion object {

        private val EVENT_LIFETIME = Duration.ofHours(24)
    }
}
