package io.meiro.sdk.api

import io.meiro.sdk.event.Event
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import okhttp3.HttpUrl.Companion.toHttpUrl
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody.Companion.toRequestBody

/**
 * Networking for sending events to the Meiro API
 */
internal class MeiroEventsApi(
    endpoint: String,
    private val okHttpClient: OkHttpClient,
) {

    @OptIn(ExperimentalSerializationApi::class)
    val jsonFormat = Json { explicitNulls = false }

    private val eventMapping = ApiEventMapping()

    private val eventsUrl = endpoint.toHttpUrl().newBuilder()
        .addPathSegment("meiro_mobile")
        .build()

    suspend fun sendEvent(
        event: Event
    ) {
        withContext(Dispatchers.IO) {
            okHttpClient.newCall(
                Request.Builder()
                    .url(eventsUrl)
                    .post(
                        jsonFormat.encodeToString(eventMapping.toApiEvent(event)).toRequestBody(
                            "application/json".toMediaType()
                        )
                    )
                    .build()
            ).execute()
        }
    }
}
