package io.meiro.sdk.track

import android.app.Activity
import android.app.Application
import android.os.Bundle
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import androidx.fragment.app.FragmentManager

/**
 * Registers an [Application.ActivityLifecycleCallbacks] that automatically tracks screen views
 * of [Activity] and [Fragment] instances.
 */
internal class AppActivityLifecycleCallbacks(
    private val onScreenViewed: (String) -> Unit
) : Application.ActivityLifecycleCallbacks {

    private val fragmentLifecycleCallbacksByActivity = mutableMapOf<Activity, FragmentManager.FragmentLifecycleCallbacks>()

    override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
        if (activity !is FragmentActivity) return
        val fragmentLifecycleCallbacks = AppFragmentLifecycleCallbacks()
        activity.supportFragmentManager.registerFragmentLifecycleCallbacks(
            fragmentLifecycleCallbacks, true
        )
        fragmentLifecycleCallbacksByActivity[activity] = fragmentLifecycleCallbacks
    }

    override fun onActivityResumed(activity: Activity) {
        onScreenViewed(activity::class.java.simpleName)
    }

    override fun onActivityDestroyed(activity: Activity) {
        if (activity !is FragmentActivity) return
        fragmentLifecycleCallbacksByActivity[activity]?.let { callbacks ->
            activity.supportFragmentManager.unregisterFragmentLifecycleCallbacks(callbacks)
            fragmentLifecycleCallbacksByActivity.remove(activity)
        }
    }

    override fun onActivityStarted(activity: Activity) = Unit
    override fun onActivityPaused(activity: Activity) = Unit
    override fun onActivityStopped(activity: Activity) = Unit
    override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) = Unit

    private inner class AppFragmentLifecycleCallbacks : FragmentManager.FragmentLifecycleCallbacks() {

        override fun onFragmentResumed(fm: FragmentManager, f: Fragment) {
            onScreenViewed(f::class.java.simpleName)
        }
    }
}
