package io.meiro.sdk.session

import java.time.Duration
import java.time.Instant
import java.util.UUID

/**
 * Manages the session of the SDK.
 */
internal class SessionManager {

    private var sessionId: String? = null

    private var appBackgroundTime: Instant? = null
    private var lastEventSentTime: Instant? = null

    fun getSessionId(now: Instant = Instant.now()): String {
        val localSessionId = sessionId ?: return generateAndSaveSessionId()
        val localAppBackgroundTime = appBackgroundTime
        if (localAppBackgroundTime != null && now.minus(APP_BACKGROUNDED_SESSION_TIMEOUT).isAfter(localAppBackgroundTime)) {
            return generateAndSaveSessionId()
        }
        val localLastEventSentTime = lastEventSentTime
        if (localLastEventSentTime != null && now.minus(EVENT_SESSION_TIMEOUT).isAfter(localLastEventSentTime)) {
            return generateAndSaveSessionId()
        }
        return localSessionId
    }

    private fun generateAndSaveSessionId(): String {
        val generatedSessionId = UUID.randomUUID().toString()
        this.sessionId = generatedSessionId
        appBackgroundTime = null
        lastEventSentTime = null
        return generatedSessionId
    }

    fun appBackgrounded() {
        appBackgroundTime = Instant.now()
    }

    fun appForegrounded() {
        appBackgroundTime = null
    }

    fun eventSent() {
        lastEventSentTime = Instant.now()
    }

    fun clear() {
        sessionId = null
    }

    companion object {

        private val EVENT_SESSION_TIMEOUT = Duration.ofMinutes(30)
        private val APP_BACKGROUNDED_SESSION_TIMEOUT = Duration.ofSeconds(60)
    }
}
