package io.meiro.sdk.audience

import io.meiro.sdk.MeiroSdk
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import okhttp3.HttpUrl
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import org.json.JSONObject

internal class MeiroAudienceImpl(
    private val okHttpClient: OkHttpClient,
    private val scope: CoroutineScope
) : MeiroAudience {

    override fun wbs(
        instance: String,
        parameters: Map<String, String>,
        segment: Int?,
        callback: MeiroAudience.ResultCallback
    ) {
        scope.launch {
            val url = MeiroAudienceUrlCreator.buildUrl(
                userId = MeiroSdk.userId,
                instance = instance,
                segment = segment,
                parameters = parameters
            )
            try {
                val response = callRequest(url)
                processResponse(response, callback)
            } catch (e: Exception) {
                callback.onError(e)
            }
        }
    }

    private fun processResponse(response: Response, callback: MeiroAudience.ResultCallback) {
        val stringResponse = response.body?.string()
        if (stringResponse == null) {
            callback.onError(MeiroAudienceError("Request failed with code ${response.code}", response.code))
        } else {
            try {
                val jsonObject = JSONObject(stringResponse)
                if (jsonObject.optString(STATUS_KEY) == "ok") {
                    val returnedAttributes = jsonObject.getJSONObject(RETURNED_ATTRIBUTES_KEY)
                    val data = jsonObject.getJSONObject(DATA_KEY)
                    callback.onSuccess(MeiroAudienceResult(returnedAttributes, data))
                } else {
                    callback.onError(MeiroAudienceError(jsonObject.optString(MESSAGE_KEY), response.code))
                }
            } catch (e: Exception) {
                callback.onError(e)
            }
        }
    }

    private suspend fun callRequest(url: HttpUrl): Response {
        return withContext(Dispatchers.IO) {
            okHttpClient.newCall(
                Request.Builder()
                    .url(url)
                    .get()
                    .build()
            ).execute()
        }
    }

    companion object {

        private const val STATUS_KEY = "status"
        private const val DATA_KEY = "data"
        private const val RETURNED_ATTRIBUTES_KEY = "returned_attributes"
        private const val MESSAGE_KEY = "message"
    }
}
