package io.meiro.sdk

import android.app.Application
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.ProcessLifecycleOwner
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import io.meiro.sdk.api.MeiroEventsApi
import io.meiro.sdk.audience.MeiroAudience
import io.meiro.sdk.audience.MeiroAudienceImpl
import io.meiro.sdk.event.Event
import io.meiro.sdk.event.EventAppInfo
import io.meiro.sdk.event.EventDeviceInfo
import io.meiro.sdk.event.EventFirebaseInfo
import io.meiro.sdk.event.EventOsInfo
import io.meiro.sdk.event.EventType
import io.meiro.sdk.event.EventUserInfo
import io.meiro.sdk.offline.SyncManager
import io.meiro.sdk.preferences.MeiroPreferences
import io.meiro.sdk.session.SessionManager
import io.meiro.sdk.track.AppActivityLifecycleCallbacks
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import java.time.Instant
import java.util.Locale

/**
 * Internal implementation of [MeiroEvents]
 */
internal class MeiroEventsImpl : MeiroEvents {

    private var logger: MeiroLogger = MeiroAndroidLogger()
    private val sdkScope = CoroutineScope(Dispatchers.Main)
    private var preferences: MeiroPreferences? = null
    private val sessionManager = SessionManager()
    private var syncManager: SyncManager? = null
    private var audience: MeiroAudience? = null

    private var configuration: Configuration? = null
    private var networking: MeiroEventsApi? = null
    private var enabled: Boolean = true
    private var context: Context? = null

    private var okHttpClient: OkHttpClient? = null

    private var googleAdId: String? = null

    override val sessionId: String
        get() = sessionManager.getSessionId()

    override val userId: String
        get() = requirePreferences().userId

    override fun init(context: Context, configuration: Configuration, logger: MeiroLogger) {
        this.configuration = configuration
        this.context = context
        this.logger = logger

        okHttpClient = OkHttpClient.Builder()
            .addInterceptor(
                HttpLoggingInterceptor { message -> logger.log(message) }.apply {
                    level = if (configuration.debugMode) HttpLoggingInterceptor.Level.BODY else HttpLoggingInterceptor.Level.NONE
                }
            )
            .build()
        networking = MeiroEventsApi(
            endpoint = configuration.endpoint,
            okHttpClient = okHttpClient!!
        )
        syncManager = SyncManager(context, networking!!, logger)
        preferences = MeiroPreferences(context, onUserIdGenerated = { fcmToken ->
            if (fcmToken != null) {
                trackEventInternal(EventType.FCM_TOKEN_REGISTERED)
            }
        })
        audience = MeiroAudienceImpl(okHttpClient!!, sdkScope)

        if (configuration.automaticTrackingEnabled) {
            setupAutomaticTracking(context)
        }
    }

    private fun setupAutomaticTracking(context: Context) {
        setupAppLifecycleTracking()
        trackAppInstalledIfNecessary()
        setupScreenTracking(context)
        resolveGoogleAdId()
    }

    private fun resolveGoogleAdId() {
        if (requireContext().checkSelfPermission("com.google.android.gms.permission.AD_ID") != PackageManager.PERMISSION_GRANTED) {
            return
        }
        sdkScope.launch(Dispatchers.IO) {
            try {
                val adIdInfo = AdvertisingIdClient.getAdvertisingIdInfo(requireContext())
                googleAdId = adIdInfo.id
            } catch (e: Exception) {
                logger.log("Failed to get Google Ad ID", e)
            }
        }
    }

    private fun setupAppLifecycleTracking() {
        ProcessLifecycleOwner.get().lifecycle.addObserver(object : DefaultLifecycleObserver {

            override fun onStart(owner: LifecycleOwner) {
                trackEventInternal(EventType.APP_FOREGROUND)
                sessionManager.appForegrounded()
            }

            override fun onStop(owner: LifecycleOwner) {
                trackEventInternal(EventType.APP_BACKGROUND)
                sessionManager.appBackgrounded()
            }
        })
    }

    private fun trackAppInstalledIfNecessary() {
        if (requirePreferences().freshInstall) {
            trackEventInternal(EventType.APP_INSTALLED)
            requirePreferences().freshInstall = false
        }
    }

    private fun setupScreenTracking(context: Context) {
        (context.applicationContext as Application).registerActivityLifecycleCallbacks(
            AppActivityLifecycleCallbacks { screenName ->
                trackScreenView(screenName)
            }
        )
    }

    override fun setEnabled(enabled: Boolean) {
        this.enabled = enabled
    }

    override fun trackCustomEvent(properties: Map<String, Any>) {
        if (!enabled) return
        trackEventInternal(EventType.CUSTOM, properties)
    }

    internal fun trackEventInternal(
        eventType: EventType,
        properties: Map<String, Any> = emptyMap(),
    ) {
        sdkScope.launch {
            val event = Event(
                type = eventType,
                properties = properties,
                timestamp = Instant.now(),
                appInfo = resolveAppInfo(),
                osInfo = resolveOsInfo(),
                deviceInfo = resolveDeviceInfo(),
                firebaseInfo = resolveFirebaseInfo(),
                userInfo = resolveUserInfo()
            )
            try {
                networking?.sendEvent(event)
                logger.log("Event $eventType sent")
                sessionManager.eventSent()
            } catch (e: Exception) {
                logger.log("Event $eventType sent failed", e)
                requireSyncManager().saveEvent(event)
            }
        }
    }

    private fun resolveAppInfo(): EventAppInfo {
        return EventAppInfo(
            id = configuration!!.appId,
            name = getApplicationName(),
            version = getApplicationVersionName(),
            language = Locale.getDefault().language,
            adId = googleAdId
        )
    }

    private fun getApplicationName(): String {
        return requireContext().packageManager.getApplicationLabel(
            requireContext().applicationInfo
        ).toString()
    }

    private fun getApplicationVersionName(): String {
        return requireContext().packageManager.getPackageInfo(
            requireContext().packageName, 0
        ).versionName
    }

    private fun resolveOsInfo(): EventOsInfo {
        return EventOsInfo(
            type = "android",
            version = Build.VERSION.RELEASE
        )
    }

    private fun resolveDeviceInfo(): EventDeviceInfo {
        return EventDeviceInfo(
            manufacturer = Build.MANUFACTURER,
            model = Build.MODEL
        )
    }

    private fun resolveUserInfo(): EventUserInfo {
        return EventUserInfo(
            userId = userId,
            sessionId = sessionId
        )
    }

    private fun resolveFirebaseInfo(): EventFirebaseInfo {
        return EventFirebaseInfo(
            projectId = requireConfiguration().firebaseProjectId,
            token = requirePreferences().fcmToken
        )
    }

    override fun resetIdentity() {
        if (!enabled) return
        requirePreferences().resetIdentity()
        sessionManager.clear()
    }

    override fun trackLinkClick(url: String) {
        if (!enabled) return
        trackEventInternal(EventType.LINK_CLICK, mapOf("url" to url))
    }

    override fun trackScreenView(name: String, properties: Map<String, Any>) {
        if (!enabled) return
        trackEventInternal(EventType.SCREEN_VIEW, properties + mapOf("name" to name))
    }

    override fun setFcmToken(token: String) {
        if (!enabled) return
        requirePreferences().fcmToken = token
        trackEventInternal(EventType.FCM_TOKEN_REGISTERED)
    }

    override fun getAudience(): MeiroAudience {
        return requireAudience()
    }

    internal suspend fun performSync() {
        logger.log("Performing sync")
        syncManager?.sync()
    }

    private fun requireContext(): Context {
        return requireNotNull(context) { "MeiroSdk.init must be called before using the SDK" }
    }

    internal fun requireConfiguration(): Configuration {
        return requireNotNull(configuration) { "MeiroSdk.init must be called before using the SDK" }
    }

    private fun requirePreferences(): MeiroPreferences {
        return requireNotNull(preferences) { "MeiroSdk.init must be called before using the SDK" }
    }

    private fun requireSyncManager(): SyncManager {
        return requireNotNull(syncManager) { "MeiroSdk.init must be called before using the SDK" }
    }

    private fun requireAudience(): MeiroAudience {
        return requireNotNull(audience) { "MeiroSdk.init must be called before using the SDK" }
    }

    fun log(message: String, exception: Throwable? = null) {
        logger.log(message, exception)
    }
}
