package io.meiro.sdk.preferences

import android.content.Context
import java.util.UUID

/**
 * Preferences of the SDK
 */
internal class MeiroPreferences(
    context: Context,
    private val onUserIdGenerated: (fcmToken: String?) -> Unit,
) {

    private val sharedPreferences = context.getSharedPreferences(PREFERENCES_NAME, Context.MODE_PRIVATE)

    var userId: String
        get() = sharedPreferences.getString(USER_ID_KEY, null).let { userId ->
            if (userId == null) {
                val generatedUserId = UUID.randomUUID().toString()
                this.userId = generatedUserId.also {
                    onUserIdGenerated(fcmToken)
                }
                generatedUserId
            } else {
                userId
            }
        }
        set(value) = sharedPreferences.edit().putString(USER_ID_KEY, value).apply()

    var fcmToken: String?
        get() = sharedPreferences.getString(FCM_TOKEN_KEY, null)
        set(value) = sharedPreferences.edit().putString(FCM_TOKEN_KEY, value).apply()

    var freshInstall: Boolean
        get() = sharedPreferences.getBoolean(FRESH_INSTALL_KEY, true)
        set(value) = sharedPreferences.edit().putBoolean(FRESH_INSTALL_KEY, value).apply()

    fun resetIdentity() {
        sharedPreferences.edit()
            .remove(USER_ID_KEY)
            .apply()
    }

    companion object {

        private const val PREFERENCES_NAME = "io.meiro.sdk.preferences"
        private const val USER_ID_KEY = "user_uuid"
        private const val FCM_TOKEN_KEY = "fcm_token"
        private const val FRESH_INSTALL_KEY = "fresh_install"
    }
}
