@file:OptIn(ExperimentalCoroutinesApi::class)

package io.meiro.sdk.audience

import androidx.annotation.MainThread
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.suspendCancellableCoroutine
import org.json.JSONObject
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

/**
 * Meiro Audience API client. Encapsulates the logic of sending requests to the Meiro Audience WBS API.
 */
interface MeiroAudience {

    /**
     * Sends a WBS request to the Meiro Audience API.
     */
    fun wbs(
        instance: String,
        parameters: Map<String, String> = emptyMap(),
        segment: Int? = null,
        callback: ResultCallback
    )

    suspend fun wbs(
        instance: String,
        parameters: Map<String, String> = emptyMap(),
        segment: Int? = null,
    ): MeiroAudienceResult {
        return suspendCancellableCoroutine {
            wbs(instance, parameters, segment, object : ResultCallback {
                override fun onSuccess(result: MeiroAudienceResult) {
                    it.resume(result)
                }

                override fun onError(error: Throwable) {
                    it.resumeWithException(error)
                }
            })
        }
    }

    /**
     * Callback interface for the Audience request. Methods are called on a main thread.
     */
    interface ResultCallback {

        @MainThread
        fun onSuccess(result: MeiroAudienceResult)

        @MainThread
        fun onError(error: Throwable)
    }
}

/**
 * Represents the result of the Audience request. Since the nature of the result may vary, the
 * generic `JSONObject` entities are returned for `returned_attributes` and `data` fields of the response.
 */
data class MeiroAudienceResult(
    val returnedAttributes: JSONObject,
    val data: JSONObject
)

/**
 * Represents an error that occurred during the Audience request.
 */
data class MeiroAudienceError(
    override val message: String,
    val code: Int
) : Throwable()
