/*
 * Copyright 2019 Netflix, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.netflix.mantis.discovery.proto;

import java.net.InetSocketAddress;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;


public class MantisWorker {

    private final String host;
    private final int port;

    @JsonCreator
    @JsonIgnoreProperties(ignoreUnknown = true)
    public MantisWorker(@JsonProperty("host") String host,
                        @JsonProperty("port") int port) {
        this.host = host;
        this.port = port;
    }

    public String getHost() {
        return host;
    }

    public int getPort() {
        return port;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        final MantisWorker that = (MantisWorker) o;
        return port == that.port
                && Objects.equals(host, that.host);
    }

    @Override
    public int hashCode() {
        return Objects.hash(host, port);
    }

    @Override
    public String toString() {
        return "MantisWorker{" + "host='" + host + '\''
                + ", port=" + port
                + '}';
    }

    /**
     * Returns an {@link InetSocketAddress} representation of this worker.
     */
    public InetSocketAddress toInetSocketAddress() {
        return InetSocketAddress.createUnresolved(host, port);
    }

    /**
     * Estimate the size (in Bytes) of this object using the size of its fields.
     */
    public int size() {
        return host.getBytes().length   // host
                + Integer.BYTES;        // port
    }
}
