package io.manbang.ebatis.core.request;

import io.manbang.ebatis.core.generic.GenericType;
import io.manbang.ebatis.core.meta.MethodMeta;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.ActionRequest;

import java.lang.annotation.Annotation;
import java.util.function.Consumer;

/**
 * @author 章多亮
 * @since 2019/12/17 15:32
 */
public abstract class AbstractRequestFactory<A extends Annotation, R extends ActionRequest> implements RequestFactory<R> {
    private final Class<? extends Annotation> requestAnnotationClass;

    @SuppressWarnings("unchecked")
    protected AbstractRequestFactory() {
        this.requestAnnotationClass = (Class<? extends Annotation>) GenericType.forType(this.getClass()).as(AbstractRequestFactory.class).resolveGeneric(0);
    }

    @Override
    public final R create(MethodMeta meta, Object... args) {
        R request = doCreate(meta, args);
        Annotation requestAnnotation = meta.getRequestAnnotation();
        if (requestAnnotationClass.isInstance(requestAnnotation)) {
            setAnnotationMeta(request, meta.getRequestAnnotation());
        }
        return request;
    }

    protected void setTypeIfNecessary(MethodMeta meta, Object[] args, Consumer<String> consumer) {
        if (StringUtils.isNotBlank(meta.getType(meta, args))) {
            consumer.accept(meta.getType(meta, args));
        }
    }

    protected void setTypesIfNecessary(MethodMeta meta, Object[] args, Consumer<String[]> consumer) {
        if (ArrayUtils.isNotEmpty(meta.getTypes(meta, args))) {
            consumer.accept(meta.getTypes(meta, args));
        }
    }


    /**
     * 设置可选信息，通过注解传递过来
     *
     * @param request    请求
     * @param annotation 注解
     */
    protected abstract void setAnnotationMeta(R request, A annotation);

    /**
     * 实际构建查询请求
     *
     * @param meta 方法元数据
     * @param args 查询条件
     * @return 请求实例
     */
    protected abstract R doCreate(MethodMeta meta, Object[] args);
}
