package io.lsn.spring.limiter.provider.key;

import io.lsn.spring.limiter.limited.LimitedInformationProvider;
import org.aspectj.lang.ProceedingJoinPoint;
import org.springframework.core.LocalVariableTableParameterNameDiscoverer;
import org.springframework.core.ParameterNameDiscoverer;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;

import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * @author Patryk Szlagowski <patryksz@lsnova.pl>
 */
public class KeyBuilder {

    /**
     * return key
     *
     * @param method
     * @return
     */
    public static String build(Method method, ProceedingJoinPoint point, String managerName) {
        String key = buildFromExpression(method, point);
        if (key != null) {
            return concatWith(key, managerName);
        }

        if (LimitedInformationProvider.isAsync(method)) {
            return concatWith(buildForAsync(method), managerName);
        }
        return concatWith(build(method), managerName);
    }

    /**
     * build key for the current method
     *
     * @param method
     * @return
     */
    public static String build(Method method) {
        String hostname = "";
        try {
            hostname = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            e.printStackTrace();
        }
        return Arrays.asList(hostname, method.getDeclaringClass().getName(), method.getName(), String.valueOf(method.hashCode()))
                .stream()
                .filter(Objects::nonNull)
                .collect(Collectors.joining("-"));
    }

    /**
     * build key for async
     * @param method
     * @return
     */
    public static String buildForAsync(Method method) {
        return Arrays.asList(method.getDeclaringClass().getName(), method.getName(), String.valueOf(method.hashCode()))
                .stream()
                .filter(Objects::nonNull)
                .collect(Collectors.joining("-"));
    }

    /**
     * build key based on keyExpression
     *
     * @param method
     * @return
     */
    public static String buildFromExpression(Method method, ProceedingJoinPoint point) {
        String expression = LimitedInformationProvider.keyExpression(method);
        if (expression == null || expression.isEmpty()) {
            return null;
        }

        Object[] args = point.getArgs();

        ParameterNameDiscoverer discoverer = new LocalVariableTableParameterNameDiscoverer();
        List<String> parameterNames = Arrays.asList(discoverer.getParameterNames(method)).stream().collect(Collectors.toList());
        Map<String, Object> parameters = IntStream
                .range(0, args.length)
                .boxed()
                .collect(Collectors.toMap(i -> parameterNames.get(i), i -> args[i]));

        ExpressionParser parser = new SpelExpressionParser();
        Expression exp = parser.parseExpression(expression);

        EvaluationContext context = new StandardEvaluationContext();
        parameters.keySet().stream().forEach(key -> context.setVariable(key, parameters.get(key)));

        return exp.getValue(context, String.class);
    }

    /**
     * concat with managerName if provided
     * @param key
     * @param managerName
     * @return
     */
    private static String concatWith(String key, String managerName) {
        if (managerName == null) {
            return key;
        }

        return key.concat("-").concat(managerName);
    }
}
