package io.lsn.spring.limiter.limiter;

import io.lsn.spring.limiter.LimitExceededException;
import io.lsn.spring.limiter.configuration.EntryLimit;
import io.lsn.spring.limiter.configuration.LimiterProperties;
import io.lsn.spring.limiter.limited.LimitedInformationProvider;
import io.lsn.spring.limiter.provider.LimitProviderInterface;
import io.lsn.spring.limiter.provider.key.KeyBuilder;
import org.aspectj.lang.ProceedingJoinPoint;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.lang.reflect.Method;

/**
 * @author Patryk Szlagowski <patryksz@lsnova.pl>
 */
@Service
public class Limiter {

    private LimitProviderInterface provider;
    private LimiterProperties configuration;

    @Autowired
    public Limiter(LimitProviderInterface provider, LimiterProperties configuration) {
        this.provider = provider;
        this.configuration = configuration;
    }

    /**
     * based on configuration, try to check in every manager if it can be proceed
     * @param point
     * @param target
     * @return
     * @throws Throwable
     */
    public Object proceed(ProceedingJoinPoint point, Method target) throws Throwable {
        String key = KeyBuilder.build(target, point, null);
        EntryLimit configurationForKey = configuration.getEntries().get(key);
        if (configurationForKey == null) {
            return point.proceed();
        }

        Object toReturn = null;
        for (String managerName : configurationForKey.getLimits().keySet()) {
            toReturn = proceedSingle(point, target, managerName, toReturn);
        }
        return toReturn;
    }

    /**
     * try to process for single entry
     *
     * @param point
     * @param target
     * @param managerName
     * @param toReturn
     * @return
     * @throws Throwable
     */
    private Object proceedSingle(ProceedingJoinPoint point, Method target, String managerName, Object toReturn) throws Throwable {
        int current = provider.current(target, point, managerName);
        provider.increase(target, point, managerName);
        current++;
        if (current >= getLimit(point, target, managerName)) {
            provider.decrease(target, point, managerName);
            throw new LimitExceededException();
        }

        if (toReturn == null) {
            toReturn = point.proceed();
        }
        if (LimitedInformationProvider.isAsync(target)) {
            provider.decrease(target, point, managerName);
        }
        return toReturn;
    }

    /**
     * limit for specific managerName
     * @param point
     * @param target
     * @param managerName
     * @return
     */
    private Integer getLimit(ProceedingJoinPoint point, Method target, String managerName) {
        String key = KeyBuilder.build(target, point, null);
        if (configuration.getEntries().containsKey(key)) {
            return configuration.getEntries().get(key).getLimits().get(managerName);
        }
        return LimitedInformationProvider.limit(target);
    }
}
