/*
 * Decompiled with CFR 0.152.
 */
package io.lighty.server.util;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.lighty.core.controller.impl.config.ConfigurationException;
import io.lighty.server.config.LightyServerConfig;
import io.lighty.server.config.SecurityConfig;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LightyServerConfigUtils {
    private static final String SERVER_CONFIG_ROOT_ELEMENT_NAME = "lighty-server";
    private static final Logger LOG = LoggerFactory.getLogger(LightyServerConfigUtils.class);

    private LightyServerConfigUtils() {
    }

    public static LightyServerConfig getServerConfiguration(InputStream jsonConfigIS) throws ConfigurationException {
        LightyServerConfig lightyServerConfig;
        JsonNode configNode;
        ObjectMapper mapper = new ObjectMapper();
        try {
            configNode = mapper.readTree(jsonConfigIS);
        }
        catch (IOException e) {
            throw new ConfigurationException("Cannot deserialize Json content to Json tree nodes", (Throwable)e);
        }
        if (!configNode.has(SERVER_CONFIG_ROOT_ELEMENT_NAME)) {
            LOG.warn("Json config does not contain {} element. Using defaults.", (Object)SERVER_CONFIG_ROOT_ELEMENT_NAME);
            return new LightyServerConfig();
        }
        JsonNode lightyServerNode = configNode.path(SERVER_CONFIG_ROOT_ELEMENT_NAME);
        try {
            lightyServerConfig = (LightyServerConfig)mapper.treeToValue((TreeNode)lightyServerNode, LightyServerConfig.class);
            lightyServerConfig.setSecurityConfig(LightyServerConfigUtils.createSecurityConfig(lightyServerConfig));
        }
        catch (JsonProcessingException e) {
            throw new ConfigurationException(String.format("Cannot bind Json tree to type: %s", LightyServerConfig.class), (Throwable)e);
        }
        return lightyServerConfig;
    }

    public static LightyServerConfig getDefaultLightyServerConfig() throws ConfigurationException {
        LightyServerConfig lightyServerConfig = new LightyServerConfig();
        lightyServerConfig.setSecurityConfig(LightyServerConfigUtils.createSecurityConfig(lightyServerConfig));
        return lightyServerConfig;
    }

    public static SecurityConfig createSecurityConfig(LightyServerConfig config) throws ConfigurationException {
        try {
            KeyStore.PasswordProtection passProtection = new KeyStore.PasswordProtection(config.getKeyStorePassword().toCharArray());
            KeyStore keyStore = KeyStore.Builder.newInstance(config.getKeyStoreType(), null, passProtection).getKeyStore();
            Optional<InputStream> ksFile = LightyServerConfigUtils.readKeyStoreFile(config.getKeyStoreFilePath());
            if (ksFile.isEmpty()) {
                throw new ConfigurationException("Unable to create KeyStore configuration: KeyStore file was not found on path: " + config.getKeyStoreFilePath());
            }
            KeyStore.PasswordProtection trustPassProtection = new KeyStore.PasswordProtection(config.getTrustKeyStorePassword().toCharArray());
            KeyStore trustKeyStore = KeyStore.Builder.newInstance(config.getKeyStoreType(), null, trustPassProtection).getKeyStore();
            Optional<InputStream> trustKsFile = LightyServerConfigUtils.readKeyStoreFile(config.getTrustKeyStoreFilePath());
            if (trustKsFile.isEmpty()) {
                throw new ConfigurationException("Unable to create TrustKeyStore config: KeyStore file was not found on path: " + config.getTrustKeyStoreFilePath());
            }
            keyStore.load(ksFile.get(), config.getKeyStorePassword().toCharArray());
            trustKeyStore.load(trustKsFile.get(), config.getTrustKeyStorePassword().toCharArray());
            return new SecurityConfig(keyStore, config.getKeyStorePassword(), trustKeyStore, config.getTrustKeyStorePassword(), config.isNeedClientAuth());
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new ConfigurationException("Unable to create KeyStore configuration", (Throwable)e);
        }
    }

    private static Optional<InputStream> readKeyStoreFile(String keyStoreFilePath) throws IOException {
        InputStream ksFile;
        try {
            LOG.info("Trying to load KeyStore from filesystem from path {}", (Object)keyStoreFilePath);
            ksFile = Files.newInputStream(Paths.get(keyStoreFilePath, new String[0]), new OpenOption[0]);
            LOG.info("KeyStore found on filesystem on path {}", (Object)keyStoreFilePath);
        }
        catch (NoSuchFileException e) {
            LOG.info("KeyStore not found on filesystem, looking in resources on path {}", (Object)keyStoreFilePath);
            ksFile = LightyServerConfigUtils.class.getClassLoader().getResourceAsStream(keyStoreFilePath);
            LOG.info("KeyStore found on classpath on path {}", (Object)keyStoreFilePath);
        }
        if (ksFile == null) {
            LOG.error("KeyStore was not found on path {} in filesystem or resources", (Object)keyStoreFilePath);
        }
        return Optional.ofNullable(ksFile);
    }
}

