/*
 * Decompiled with CFR 0.152.
 */
package io.leoplatform.sdk.aws.payload;

import io.leoplatform.sdk.ExecutorManager;
import io.leoplatform.sdk.config.ConnectorConfig;
import io.leoplatform.sdk.payload.ThresholdMonitor;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class InternalThresholdMonitor
implements ThresholdMonitor {
    private static final Logger log = LoggerFactory.getLogger(InternalThresholdMonitor.class);
    private final long maxBytesPerSecond;
    private final BigDecimal warningThreshold;
    private final ExecutorManager executorManager;
    private final AtomicBoolean running;
    private final Lock lock = new ReentrantLock();
    private final Condition thresholdCheck = this.lock.newCondition();
    private final AtomicLong currentLevel = new AtomicLong();
    private final AtomicBoolean failover = new AtomicBoolean(false);

    @Inject
    public InternalThresholdMonitor(ConnectorConfig config, ExecutorManager executorManager) {
        this.maxBytesPerSecond = config.longValueOrElse("Stream.BytesPerSecondFailover", Long.valueOf(50000L));
        this.warningThreshold = this.eightyPercentOfMax();
        this.executorManager = executorManager;
        this.running = new AtomicBoolean(true);
        if (this.maxBytesPerSecond > 0L) {
            CompletableFuture.runAsync(this::checkThresholds, executorManager.get());
        }
    }

    public void addBytes(Long bytes) {
        this.currentLevel.addAndGet(bytes);
        this.signalAll();
    }

    public boolean isFailover() {
        return this.failover.get();
    }

    public void end() {
        this.running.set(false);
        this.signalAll();
    }

    private void signalAll() {
        this.lock.lock();
        try {
            this.thresholdCheck.signalAll();
        }
        finally {
            this.lock.unlock();
        }
    }

    private boolean wasOverThreshold() {
        return this.currentLevel.getAndSet(0L) > this.maxBytesPerSecond;
    }

    private boolean isOverThreshold() {
        return this.currentLevel.get() > this.maxBytesPerSecond;
    }

    private void checkThresholds() {
        do {
            if (this.wasOverThreshold()) {
                boolean wasFailover = this.failover.getAndSet(true);
                if (!wasFailover) {
                    log.info("Exceeded {} bytes/second", (Object)this.maxBytesPerSecond);
                    log.warn("Failover enabled", (Object)this.maxBytesPerSecond);
                }
            } else {
                CompletableFuture.runAsync(this::delayedClearCheck, this.executorManager.get());
            }
            this.lock.lock();
            try {
                this.thresholdCheck.await(1L, TimeUnit.SECONDS);
            }
            catch (InterruptedException i) {
                this.running.set(false);
                log.info("Threshold monitor stopped");
            }
            finally {
                this.lock.unlock();
            }
        } while (this.running.get());
        this.failover.set(false);
    }

    private BigDecimal percentageOfThreshold(BigDecimal level) {
        return level.divide(new BigDecimal(this.maxBytesPerSecond), RoundingMode.HALF_EVEN).movePointRight(2).setScale(0, RoundingMode.HALF_EVEN);
    }

    private void delayedClearCheck() {
        if (this.approachingThreshold()) {
            this.thresholdWarning();
        }
        this.lock.lock();
        try {
            this.thresholdCheck.await(10L, TimeUnit.SECONDS);
            if (this.running.get() && this.failover.get() && this.isOverThreshold()) {
                log.warn("Failover remains in place");
            } else if (this.failover.get()) {
                this.failover.set(false);
                log.info("Cleared failover");
            }
        }
        catch (InterruptedException i) {
            this.running.set(false);
        }
        finally {
            this.lock.unlock();
        }
    }

    private BigDecimal eightyPercentOfMax() {
        return new BigDecimal(this.maxBytesPerSecond).multiply(new BigDecimal(".8")).setScale(0, RoundingMode.HALF_UP);
    }

    private boolean approachingThreshold() {
        BigDecimal level = new BigDecimal(this.currentLevel.get());
        return level.compareTo(this.warningThreshold) > 0;
    }

    private void thresholdWarning() {
        BigDecimal level = new BigDecimal(this.currentLevel.get());
        BigDecimal percentageOfThreshold = this.percentageOfThreshold(level);
        log.warn("Bytes per second are currently {}% of your failover threshold", (Object)percentageOfThreshold);
    }
}

