/*
 * Decompiled with CFR 0.152.
 */
package io.leoplatform.sdk.aws.s3;

import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.auth.profile.ProfileCredentialsProvider;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.amazonaws.services.s3.transfer.TransferManagerBuilder;
import com.amazonaws.services.s3.transfer.Upload;
import com.amazonaws.services.s3.transfer.model.UploadResult;
import io.leoplatform.sdk.ExecutorManager;
import io.leoplatform.sdk.StreamStats;
import io.leoplatform.sdk.aws.s3.PendingS3Upload;
import io.leoplatform.sdk.aws.s3.S3Payload;
import io.leoplatform.sdk.aws.s3.S3Results;
import io.leoplatform.sdk.bus.LoadingBot;
import io.leoplatform.sdk.config.ConnectorConfig;
import java.time.Duration;
import java.time.Instant;
import java.util.LinkedList;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class S3TransferManager {
    private static final Logger log = LoggerFactory.getLogger(S3TransferManager.class);
    private final String name;
    private final TransferManager s3TransferManager;
    private final S3Results resultsProcessor;
    private final LoadingBot bot;
    private final Queue<PendingS3Upload> pendingUploads = new LinkedList<PendingS3Upload>();
    private final AtomicBoolean running;
    private final AtomicBoolean uploading;
    private final Lock lock = new ReentrantLock();
    private final Condition newUpload = this.lock.newCondition();

    @Inject
    public S3TransferManager(ConnectorConfig config, ExecutorManager executorManager, S3Results resultsProcessor, LoadingBot bot) {
        this.name = config.value("Storage.Name");
        this.s3TransferManager = TransferManagerBuilder.standard().withS3Client(this.client(config.valueOrElse("AwsProfile", ""))).withDisableParallelDownloads(Boolean.valueOf(false)).build();
        this.resultsProcessor = resultsProcessor;
        this.bot = bot;
        this.running = new AtomicBoolean(true);
        this.uploading = new AtomicBoolean(false);
        CompletableFuture.runAsync(this::synchronousUpload, executorManager.get());
    }

    void enqueue(PendingS3Upload pendingUpload) {
        if (this.running.get()) {
            this.lock.lock();
            try {
                this.pendingUploads.add(pendingUpload);
                this.newUpload.signalAll();
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    void flush() {
        this.signalUploader();
        this.awaitUploader();
    }

    private void synchronousUpload() {
        while (this.running.get()) {
            this.lock.lock();
            try {
                if (this.pendingUploads.isEmpty()) {
                    this.uploading.set(false);
                    this.newUpload.await();
                    this.uploading.set(true);
                }
            }
            catch (InterruptedException i) {
                this.running.set(false);
                this.pendingUploads.clear();
                log.warn("S3 transfer manager stopped with {} pending", (Object)this.pendingUploads.size());
            }
            finally {
                this.lock.unlock();
            }
            while (!this.pendingUploads.isEmpty()) {
                this.uploadNext();
            }
        }
        this.uploading.set(false);
        log.info("Transfer manager stopped");
    }

    private void uploadNext() {
        PendingS3Upload next;
        this.lock.lock();
        try {
            next = this.pendingUploads.remove();
        }
        catch (Exception e) {
            log.warn("Unexpectedly empty upload queue");
            return;
        }
        finally {
            this.lock.unlock();
        }
        this.upload(next);
    }

    private void upload(PendingS3Upload next) {
        log.info("Beginning upload of {} to S3", (Object)next.filename());
        PutObjectRequest request = next.s3PutRequest(this.name);
        Upload upload = this.s3TransferManager.upload(request);
        try {
            UploadResult uploadResult = upload.waitForUploadResult();
            S3Payload s3Payload = next.s3Payload(uploadResult, this.bot.name());
            log.info("{} byte upload of {} complete", (Object)s3Payload.getGzipSize(), (Object)next.filename());
            this.resultsProcessor.addSuccess(s3Payload, uploadResult);
        }
        catch (Exception e) {
            log.warn("S3 upload unexpectedly stopped");
            this.running.set(false);
            this.resultsProcessor.addFailure(upload.getDescription(), e);
        }
    }

    StreamStats end() {
        this.flush();
        this.running.set(false);
        this.signalUploader();
        this.s3TransferManager.shutdownNow();
        return this.getStats();
    }

    private void awaitUploader() {
        while (!this.pendingUploads.isEmpty() || this.uploading.get()) {
            this.signalUploader();
            this.lock.lock();
            try {
                this.newUpload.await(100L, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                log.warn("S3 transfer manager unexpectedly stopped");
                this.pendingUploads.clear();
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    private void signalUploader() {
        this.lock.lock();
        try {
            this.newUpload.signalAll();
        }
        finally {
            this.lock.unlock();
        }
    }

    private AmazonS3 client(String awsProfile) {
        return (AmazonS3)((AmazonS3ClientBuilder)AmazonS3ClientBuilder.standard().withCredentials(this.credentials(awsProfile))).build();
    }

    private AWSCredentialsProvider credentials(String awsProfile) {
        return Optional.of(awsProfile).map(String::trim).filter(profile -> !profile.isEmpty()).map(ProfileCredentialsProvider::new).map(AWSCredentialsProvider.class::cast).orElse((AWSCredentialsProvider)DefaultAWSCredentialsProviderChain.getInstance());
    }

    private StreamStats getStats() {
        return new StreamStats(){

            public Long successes() {
                return S3TransferManager.this.resultsProcessor.successes();
            }

            public Long failures() {
                return S3TransferManager.this.resultsProcessor.failures();
            }

            public Duration totalTime() {
                return Duration.between(S3TransferManager.this.resultsProcessor.start(), Instant.now());
            }
        };
    }
}

